package ru.yandex.direct.core.entity.campaign.service.validation.type.bean;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.campaign.model.BrandSurveyStatus;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithBrandLift;
import ru.yandex.direct.core.entity.campaign.model.SurveyStatus;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants;
import ru.yandex.direct.metrika.client.model.response.RetargetingCondition;
import ru.yandex.direct.validation.Predicates;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.BRAND_LIFT_ID_MAX_LENGTH;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.BRAND_LIFT_ID_MIN_LENGTH;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.brandLiftExperimentSegmentsCantBeChanged;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicateOfNullable;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.maxListSize;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;

public class CampaignWithBrandLiftValidator implements Validator<CampaignWithBrandLift, Defect> {

    private final Map<Long, List<Long>> retargetingConditionIdToGoalIds;
    private final Map<Long, BrandSurveyStatus> brandStatusByCampaignId;
    private final List<RetargetingCondition> clientAbSegments;
    private final Map<Long, Boolean> isBrandLiftHiddenByCampaignId;
    private final Map<Long, String> brandSurveyIdByCampaignId;

    public CampaignWithBrandLiftValidator(Map<Long, List<Long>> retargetingConditionIdToGoalIds,
                                          Map<Long, BrandSurveyStatus> brandStatusByCampaignId,
                                          List<RetargetingCondition> clientAbSegments,
                                          Map<Long, Boolean> isBrandLiftHiddenByCampaignId,
                                          Map<Long, String> brandSurveyIdByCampaignId) {
        this.retargetingConditionIdToGoalIds = retargetingConditionIdToGoalIds;
        this.brandStatusByCampaignId = brandStatusByCampaignId;
        this.clientAbSegments = clientAbSegments;
        this.isBrandLiftHiddenByCampaignId = isBrandLiftHiddenByCampaignId;
        this.brandSurveyIdByCampaignId = brandSurveyIdByCampaignId;
    }

    @Override
    public ValidationResult<CampaignWithBrandLift, Defect> apply(CampaignWithBrandLift campaignWithBrandLift) {
        List<Long> abSegmentGoalIdsInCampaign = retargetingConditionIdToGoalIds
                .getOrDefault(campaignWithBrandLift.getAbSegmentRetargetingConditionId(), emptyList());

        Set<Long> clientSectionIds = StreamEx.of(clientAbSegments)
                .map(RetargetingCondition::getSectionId)
                .nonNull()
                .toSet();
        Set<Long> clientAbSegmentIds = listToSet(clientAbSegments,
                ru.yandex.direct.metrika.client.model.response.RetargetingCondition::getId);

        boolean hasBrandSurveyId = campaignWithBrandLift.getBrandSurveyId() != null;
        boolean isBrandLiftHiddenOld = isBrandLiftHiddenByCampaignId.getOrDefault(campaignWithBrandLift.getId(), false);
        boolean isBrandLiftHiddenNew = nvl(campaignWithBrandLift.getIsBrandLiftHidden(), false);
        boolean brandLiftChangedToNormal = isBrandLiftHiddenOld && !isBrandLiftHiddenNew;
        boolean brandLiftChangedToHidden = !isBrandLiftHiddenOld && isBrandLiftHiddenNew;

        var vb = ModelItemValidationBuilder.of(campaignWithBrandLift);
        vb.item(CampaignWithBrandLift.BRAND_SURVEY_ID)
                .check(fromPredicate(Predicates.length(BRAND_LIFT_ID_MIN_LENGTH, BRAND_LIFT_ID_MAX_LENGTH),
                        CommonDefects.validId()));

        vb.list(CampaignWithBrandLift.AB_SEGMENT_GOAL_IDS)
                .check(newAbSegmentGoalIdsContainsOld(abSegmentGoalIdsInCampaign),
                        When.isTrue(hasBrandSurveyId && !brandLiftChangedToNormal))
                .checkEach(inSet(clientAbSegmentIds),
                        When.isValidAnd(When.isFalse(isBrandLiftHiddenNew)))
                .check(fromPredicate(Objects::isNull, brandLiftExperimentSegmentsCantBeChanged()),
                        When.isValidAnd(When.isTrue(brandLiftChangedToHidden)));

        vb.list(CampaignWithBrandLift.SECTION_IDS)
                .checkEach(inSet(clientSectionIds),
                        When.isValidAnd(When.isFalse(isBrandLiftHiddenNew)))
                .check(maxListSize(CampaignConstants.MAX_EXPERIMENTS_COUNT))
                .check(fromPredicate(sectionIds -> sectionIds == null || sectionIds.isEmpty(),
                        brandLiftExperimentSegmentsCantBeChanged()),
                        When.isTrue(brandLiftChangedToHidden));

        return vb.getResult();
    }

    // В новом списке должны содержаться цели из существующего, есть подключен BrandLift DIRECT-97618
    private static Constraint<List<Long>, Defect> newAbSegmentGoalIdsContainsOld(List<Long> existedAbSegmentGoalIds) {
        return fromPredicateOfNullable(newIds -> new HashSet<>(nvl(newIds, emptyList())).containsAll(existedAbSegmentGoalIds),
                brandLiftExperimentSegmentsCantBeChanged());
    }

    private static Constraint<String, Defect> cantBeNullIfStatusNotDraft(SurveyStatus surveyStatus) {
        return fromPredicateOfNullable(brandSurveyId -> brandSurveyId != null || surveyStatus == SurveyStatus.DRAFT,
                CommonDefects.notNull());
    }
}
