package ru.yandex.direct.core.entity.campaign.service.validation.type.bean;

import java.util.Optional;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.campaign.model.BroadMatch;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithBroadMatch;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.BROAD_MATCH_LIMIT_MAX;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.BROAD_MATCH_LIMIT_MIN;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.relevantKeywordsUselessForChosenStrategy;
import static ru.yandex.direct.core.entity.campaign.service.validation.type.bean.CampaignWithBroadMatchValidator.checkExistence;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.NumberConstraints.inRange;
import static ru.yandex.direct.validation.defect.CommonDefects.validId;

@ParametersAreNonnullByDefault
public class CampaignWithBroadMatchBeforeApplyValidator implements Validator<ModelChanges<CampaignWithBroadMatch>,
        Defect> {
    private final Set<Long> clientGoalIds;
    private final CampaignWithBroadMatch currentCampaign;

    private CampaignWithBroadMatchBeforeApplyValidator(Set<Long> clientGoalIds,
                                                       CampaignWithBroadMatch currentCampaign) {
        this.clientGoalIds = clientGoalIds;
        this.currentCampaign = currentCampaign;
    }

    public static CampaignWithBroadMatchBeforeApplyValidator build(
            Set<Long> clientGoalIds,
            CampaignWithBroadMatch currentCampaign) {
        return new CampaignWithBroadMatchBeforeApplyValidator(clientGoalIds, currentCampaign);
    }

    @Override
    public ValidationResult<ModelChanges<CampaignWithBroadMatch>, Defect> apply(
            ModelChanges<CampaignWithBroadMatch> campaignChanges) {
        var vb = ItemValidationBuilder.of(campaignChanges, Defect.class);
        BroadMatch broadMatch = campaignChanges.getPropIfChanged(CampaignWithBroadMatch.BROAD_MATCH);
        DbStrategy strategy = campaignChanges.getPropIfChanged(CampaignWithBroadMatch.STRATEGY);
        if (broadMatch == null && strategy == null) {
            return vb.getResult();
        }

        Boolean broadMatchFlag = nvl(broadMatch, currentCampaign.getBroadMatch()).getBroadMatchFlag();
        Boolean isSearchStop = nvl(strategy, currentCampaign.getStrategy()).isSearchStop();
        // Предупреждаем, что ДРФ будет выключено (в CampaignWithBroadMatchUpdateOperationSupport)
        vb.item(broadMatchFlag, BroadMatch.BROAD_MATCH_FLAG.name())
                .weakCheck(fromPredicate(flag -> !(flag && isSearchStop), relevantKeywordsUselessForChosenStrategy()));

        if (broadMatch == null) {
            return vb.getResult();
        }

        ItemValidationBuilder<BroadMatch, Defect> broadMatchValidationBuilder =
                vb.item(broadMatch, CampaignWithBroadMatch.BROAD_MATCH.name());

        broadMatchValidationBuilder
                .item(broadMatch.getBroadMatchFlag(), BroadMatch.BROAD_MATCH_FLAG.name())
                .check(notNull());

        broadMatchValidationBuilder
                .item(broadMatch.getBroadMatchLimit(), BroadMatch.BROAD_MATCH_LIMIT.name())
                .check(notNull())
                .check(inRange(BROAD_MATCH_LIMIT_MIN, BROAD_MATCH_LIMIT_MAX), When.isTrue(nvl(broadMatchFlag, false)));

        Long currentBroadMatchGoalId = Optional.ofNullable(currentCampaign)
                .map(CampaignWithBroadMatch::getBroadMatch)
                .map(BroadMatch::getBroadMatchGoalId)
                .orElse(null);

        broadMatchValidationBuilder
                .item(broadMatch.getBroadMatchGoalId(), BroadMatch.BROAD_MATCH_GOAL_ID.name())
                // 0L валидный ID = выбраны все цели
                .check(fromPredicate(goalId -> goalId >= 0L, validId()), When.notNull())
                .check(goalId -> checkExistence(goalId, clientGoalIds), When.isTrue(
                        broadMatch.getBroadMatchGoalId() != null &&
                                !broadMatch.getBroadMatchGoalId()
                                        .equals(currentBroadMatchGoalId)));

        return vb.getResult();

    }
}
