package ru.yandex.direct.core.entity.campaign.service.validation.type.bean;

import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.campaign.model.BroadMatch;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithBroadMatch;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.BROAD_MATCH_LIMIT_MAX;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.BROAD_MATCH_LIMIT_MIN;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.NumberConstraints.inRange;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;
import static ru.yandex.direct.validation.defect.CommonDefects.validId;

@ParametersAreNonnullByDefault
public class CampaignWithBroadMatchValidator implements Validator<CampaignWithBroadMatch, Defect> {

    private final Set<Long> clientGoalIds;

    private CampaignWithBroadMatchValidator(@Nullable Set<Long> clientGoalIds) {
        this.clientGoalIds = clientGoalIds;
    }

    public static CampaignWithBroadMatchValidator build(@Nullable Set<Long> clientGoalIds) {
        return new CampaignWithBroadMatchValidator(clientGoalIds);
    }

    @Override
    public ValidationResult<CampaignWithBroadMatch, Defect> apply(CampaignWithBroadMatch campaign) {

        var vb = ModelItemValidationBuilder.of(campaign);

        if (campaign.getBroadMatch() == null) {
            return vb.getResult();
        }

        ModelItemValidationBuilder<BroadMatch> broadMatchValidationBuilder =
                vb.modelItem(CampaignWithBroadMatch.BROAD_MATCH);

        Boolean broadMatchFlag = campaign.getBroadMatch().getBroadMatchFlag();

        broadMatchValidationBuilder
                .item(BroadMatch.BROAD_MATCH_LIMIT)
                .check(notNull())
                .check(inRange(BROAD_MATCH_LIMIT_MIN, BROAD_MATCH_LIMIT_MAX), When.isTrue(nvl(broadMatchFlag, false)));

        broadMatchValidationBuilder
                .item(BroadMatch.BROAD_MATCH_GOAL_ID)
                // 0L валидный ID = выбраны все цели
                .check(Constraint.fromPredicate(goalId -> goalId >= 0L, validId()), When.notNull())
                .check(goalId -> checkExistence(goalId, clientGoalIds), When.notNull());

        return vb.getResult();
    }

    public static Defect checkExistence(Long goalId, Set<Long> clientGoalIds) {
        if (goalId == 0L) {
            return null;
        }

        if (clientGoalIds != null && clientGoalIds.contains(goalId)) {
            return null;
        }

        return objectNotFound();
    }

}
