package ru.yandex.direct.core.entity.campaign.service.validation.type.bean;

import java.time.LocalDate;
import java.util.Comparator;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.campaign.model.CampaignWithCampaignsPromotions;
import ru.yandex.direct.core.entity.campaign.model.CampaignsPromotion;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MAX_ACTIVE_CAMPAIGNS_PROMOTIONS;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MAX_CAMPAIGNS_PROMOTIONS_PERIOD;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MAX_CAMPAIGNS_PROMOTION_PERCENT;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MIN_CAMPAIGNS_PROMOTION_PERCENT;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignsPromotionsPeriodsAreIntersected;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.maxActiveCampaignsPromotions;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.DateConstraints.isNotAfterThan;
import static ru.yandex.direct.validation.constraint.DateConstraints.isNotBeforeThan;
import static ru.yandex.direct.validation.constraint.NumberConstraints.inRange;

@ParametersAreNonnullByDefault
public class CampaignWithCampaignsPromotionsValidator implements Validator<CampaignWithCampaignsPromotions, Defect> {

    @Override
    public ValidationResult<CampaignWithCampaignsPromotions, Defect> apply(CampaignWithCampaignsPromotions campaign) {
        var vb = ModelItemValidationBuilder.of(campaign);
        if (campaign.getCampaignsPromotions() == null) {
            return vb.getResult();
        }
        vb.list(CampaignWithCampaignsPromotions.CAMPAIGNS_PROMOTIONS)
                .checkEach(notNull())
                .checkEachBy(campaignsPromotion -> new CampaignsPromotionValidator().apply(campaignsPromotion),
                        When.isValid())
                .checkEach(unique(CampaignsPromotion::getPromotionId),
                        When.isValidAnd(When.valueIs(cp -> cp.getPromotionId() != 0)));
        vb.item(CampaignWithCampaignsPromotions.CAMPAIGNS_PROMOTIONS).check(maxNumerOfActivePromotionsAreNotExceeded(), When.isValid());
        vb.item(CampaignWithCampaignsPromotions.CAMPAIGNS_PROMOTIONS).check(periodsAreNotIntersected(), When.isValid());
        return vb.getResult();
    }

    private static Constraint<List<CampaignsPromotion>, Defect> periodsAreNotIntersected() {
        return campaignsPromotions -> {
            campaignsPromotions.sort(Comparator.comparing(CampaignsPromotion::getStart));
            for (int i = 0; i < campaignsPromotions.size() - 1; i++) {
                LocalDate currentFinish = campaignsPromotions.get(i).getFinish();
                LocalDate nextStart = campaignsPromotions.get(i + 1).getStart();
                if (currentFinish.compareTo(nextStart) >= 0) {
                    return campaignsPromotionsPeriodsAreIntersected();
                }
            }
            return null;
        };
    }

    private static Constraint<List<CampaignsPromotion>, Defect> maxNumerOfActivePromotionsAreNotExceeded() {
        LocalDate now = LocalDate.now();
        return campaignsPromotions ->
                filterList(campaignsPromotions, cp -> !cp.getFinish().isBefore(now))
                        .size() <= MAX_ACTIVE_CAMPAIGNS_PROMOTIONS
                        ? null
                        : maxActiveCampaignsPromotions(MAX_ACTIVE_CAMPAIGNS_PROMOTIONS);
    }

    public static class CampaignsPromotionValidator implements Validator<CampaignsPromotion, Defect> {

        public static CampaignWithCampaignsPromotionsValidator.CampaignsPromotionValidator build() {
            return new CampaignWithCampaignsPromotionsValidator.CampaignsPromotionValidator();
        }

        @Override
        public ValidationResult<CampaignsPromotion, Defect> apply(CampaignsPromotion campaignsPromotion) {
            var vb = ModelItemValidationBuilder.of(campaignsPromotion);
            vb.item(CampaignsPromotion.PROMOTION_ID)
                    .check(notNull());
            vb.item(CampaignsPromotion.CID)
                    .check(notNull());
            var vbStart = vb.item(CampaignsPromotion.START)
                    .check(notNull());
            vb.item(CampaignsPromotion.FINISH)
                    .check(notNull())
                    .check(isNotBeforeThan(campaignsPromotion.getStart()), When.isValidBoth(vbStart))
                    .check(isNotAfterThan(nvl(campaignsPromotion.getStart(), LocalDate.now())
                                    .plusDays(MAX_CAMPAIGNS_PROMOTIONS_PERIOD)),
                            When.isValidBoth(vbStart));
            vb.item(CampaignsPromotion.PERCENT)
                    .check(notNull())
                    .check(inRange(MIN_CAMPAIGNS_PROMOTION_PERCENT, MAX_CAMPAIGNS_PROMOTION_PERCENT));
            return vb.getResult();
        }
    }
}
