package ru.yandex.direct.core.entity.campaign.service.validation.type.bean;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithContentLanguage;
import ru.yandex.direct.core.entity.campaign.model.ContentLanguage;
import ru.yandex.direct.core.entity.region.AllowedRegionsForLanguageContainer;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.queryrec.model.Language;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.inconsistentCampaignLanguageWithAdGroupGeo;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.operatorCannotSetContentLanguage;
import static ru.yandex.direct.core.entity.region.RegionConstants.LANGUAGE_TO_ALLOWED_REGIONS_CONTAINER;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

@ParametersAreNonnullByDefault
public class CampaignWithContentLanguageValidator {

    private final GeoTree russianGeoTree;
    private final Map<Long, List<AdGroupSimple>> adGroupsSimpleByCampaignId;
    private final boolean hasRights;

    public CampaignWithContentLanguageValidator(
            GeoTree russianGeoTree,
            Map<Long, List<AdGroupSimple>> adGroupsSimpleByCampaignId,
            boolean hasRights
    ) {
        this.russianGeoTree = russianGeoTree;
        this.adGroupsSimpleByCampaignId = adGroupsSimpleByCampaignId;
        this.hasRights = hasRights;
    }

    public ValidationResult<ModelChanges<CampaignWithContentLanguage>, Defect> apply(ModelChanges<CampaignWithContentLanguage> campaignChanges) {
        var vb = ItemValidationBuilder.of(campaignChanges, Defect.class);

        if (!campaignChanges.isPropChanged(CampaignWithContentLanguage.CONTENT_LANGUAGE)) {
            return vb.getResult();
        }

        ContentLanguage language = campaignChanges.getPropIfChanged(CampaignWithContentLanguage.CONTENT_LANGUAGE);
        List<AdGroupSimple> adGroups = adGroupsSimpleByCampaignId.get(campaignChanges.getId());

        vb.item(language, CampaignWithContentLanguage.CONTENT_LANGUAGE.name())
                .check(fromPredicate(m -> hasRights, operatorCannotSetContentLanguage()))
                .check(checkCampaignLanguageConsistentWithAdGroupGeo(adGroups), When.isValid());

        return vb.getResult();
    }

    private Constraint<ContentLanguage, Defect> checkCampaignLanguageConsistentWithAdGroupGeo(
            @Nullable List<AdGroupSimple> adGroups) {
        return contentLanguage -> {
            if (contentLanguage == null || adGroups == null) {
                return null;
            }

            Language campaignLanguage = Language.getByName(contentLanguage.getTypedValue());
            Set<Long> allowableGeoIds = LANGUAGE_TO_ALLOWED_REGIONS_CONTAINER
                    .getOrDefault(campaignLanguage, AllowedRegionsForLanguageContainer.EMPTY).getAllowedRegionIds();
            Set<Long> adGroupsWithInconsistentGeo = StreamEx.of(adGroups)
                    .filter(adGroup -> allowableGeoIds != null
                            && !russianGeoTree.isRegionsIncludedIn(adGroup.getGeo(), allowableGeoIds))
                    .map(AdGroupSimple::getId)
                    .toSet();

            if (!adGroupsWithInconsistentGeo.isEmpty()) {
                return inconsistentCampaignLanguageWithAdGroupGeo(campaignLanguage, adGroupsWithInconsistentGeo);
            }
            return null;
        };
    }
}
