package ru.yandex.direct.core.entity.campaign.service.validation.type.bean;

import java.math.BigDecimal;

import javax.annotation.Nullable;

import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithCustomDayBudget;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstraints.dayBudgetCanBeSet;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstraints.isDayBudgetOverridenByWallet;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstraints.isDayBudgetShowModeOverridenByWallet;
import static ru.yandex.direct.utils.CommonUtils.isValidId;
import static ru.yandex.direct.utils.NumberUtils.greaterThanZero;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.NumberConstraints.inRange;

public class CampaignWithCustomDayBudgetValidator implements Validator<CampaignWithCustomDayBudget, Defect> {
    private final Currency currency;
    private final Campaign wallet;
    private final AppliedChanges<CampaignWithCustomDayBudget> appliedChanges;

    public CampaignWithCustomDayBudgetValidator(Currency currency,
                                                Campaign wallet,
                                                @Nullable AppliedChanges<CampaignWithCustomDayBudget> appliedChanges) {
        this.currency = currency;
        this.wallet = wallet;
        this.appliedChanges = appliedChanges;
    }

    @Override
    public ValidationResult<CampaignWithCustomDayBudget, Defect> apply(CampaignWithCustomDayBudget campaignWithDayBudget) {
        BigDecimal minDailyBudget = campaignWithDayBudget.getType().equals(CampaignType.WALLET) ?
                currency.getMinWalletDayBudget() : currency.getMinDayBudget();

        var vb = ModelItemValidationBuilder.of(campaignWithDayBudget);
        if (appliedChanges == null || appliedChanges.passed(CampaignWithCustomDayBudget.DAY_BUDGET)) {
            vb.item(CampaignWithCustomDayBudget.DAY_BUDGET)
                    .check(notNull())
                    .check(dayBudgetCanBeSet(campaignWithDayBudget.getDayBudgetDailyChangeCount()), When.isValid())
                    .check(inRange(minDailyBudget, currency.getMaxDailyBudgetAmount()), When.isValidAnd(When.isTrue(
                            greaterThanZero(campaignWithDayBudget.getDayBudget()))))
                    .weakCheck(isDayBudgetOverridenByWallet(wallet),
                            When.isValidAnd(When.isTrue(greaterThanZero(campaignWithDayBudget.getDayBudget())
                                    && walletHasDayBudget(campaignWithDayBudget))));
        }
        if (appliedChanges == null || appliedChanges.passed(CampaignWithCustomDayBudget.DAY_BUDGET_SHOW_MODE)) {
            vb.item(CampaignWithCustomDayBudget.DAY_BUDGET_SHOW_MODE)
                    .check(notNull())
                    .weakCheck(isDayBudgetShowModeOverridenByWallet(wallet),
                            When.isValidAnd(When.isTrue(greaterThanZero(campaignWithDayBudget.getDayBudget())
                                    && walletHasDayBudget(campaignWithDayBudget))));
        }
        return vb.getResult();
    }

    private boolean walletHasDayBudget(CampaignWithCustomDayBudget campaignWithDayBudget) {
        return greaterThanZero(campaignWithDayBudget.getDayBudget())
                && isValidId(campaignWithDayBudget.getWalletId())
                && greaterThanZero(wallet.getStrategy().getDayBudget());
    }

}
