package ru.yandex.direct.core.entity.campaign.service.validation.type.bean;

import ru.yandex.direct.core.entity.campaign.model.CampaignWithEshowsSettingsAndStrategy;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.EshowsSettings;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.videoTypeNotSupportedWithStrategy;

public class CampaignWithEshowsSettingsAndStrategyValidator implements Validator<CampaignWithEshowsSettingsAndStrategy, Defect> {
    private final Boolean validateVideoTypeWithStrategy;

    private CampaignWithEshowsSettingsAndStrategyValidator(Boolean validateVideoTypeWithStrategy) {
        this.validateVideoTypeWithStrategy = validateVideoTypeWithStrategy;
    }

    public static CampaignWithEshowsSettingsAndStrategyValidator build() {
        // При создании кампании пользователь точно причастен к заполненности videoTarget, надо валидировать
        return new CampaignWithEshowsSettingsAndStrategyValidator(true);
    }

    public static CampaignWithEshowsSettingsAndStrategyValidator build(
            AppliedChanges<CampaignWithEshowsSettingsAndStrategy> appliedChanges) {
        // Если пользователь при обновлении стратегии не передал videoTarget, то не нужно запускать валидацию,
        // т.к. videoTarget могло быть ранее автоматически заполнено в beforeExecution и там же теперь будет очищено
        // Не за чем ругать пользователя, когда он не виноват в заполненности videoTarget
        return new CampaignWithEshowsSettingsAndStrategyValidator(
                appliedChanges.changed(CampaignWithEshowsSettingsAndStrategy.ESHOWS_SETTINGS));
    }

    @Override
    public ValidationResult<CampaignWithEshowsSettingsAndStrategy, Defect> apply(
            CampaignWithEshowsSettingsAndStrategy campaign
    ) {
        var vb = ModelItemValidationBuilder.of(campaign);
        vb.item(campaign.ESHOWS_SETTINGS)
                .weakCheck(validateVideoTypeWithStrategy(campaign.getStrategy()),
                        When.isTrue(validateVideoTypeWithStrategy));
        return vb.getResult();
    }

    private Constraint<EshowsSettings, Defect> validateVideoTypeWithStrategy(DbStrategy strategy) {
        return Constraint.fromPredicate(eshowsSettings ->
                        !(isAvgCpvStrategy(strategy) && eshowsSettings.getVideoType() != null),
                videoTypeNotSupportedWithStrategy()
        );
    }

    private static boolean isAvgCpvStrategy(DbStrategy strategy) {
        StrategyName strategyName = strategy.getStrategyName();

        return strategyName.equals(StrategyName.AUTOBUDGET_AVG_CPV) ||
                strategyName.equals(StrategyName.AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD);
    }
}
