package ru.yandex.direct.core.entity.campaign.service.validation.type.bean;

import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.campaign.model.CampaignWithMeaningfulGoalsWithRequiredFields;
import ru.yandex.direct.core.entity.campaign.model.DbStrategyBase;
import ru.yandex.direct.core.entity.campaign.model.MeaningfulGoal;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.core.entity.campaign.service.validation.type.container.CampaignValidationContainer;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.metrika.client.model.response.CounterInfoDirect;
import ru.yandex.direct.metrika.client.model.response.UserCountersExtended;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.ENGAGED_SESSION_GOAL_ID;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MEANINGFUL_GOALS_MAX_COUNT;
import static ru.yandex.direct.core.entity.campaign.service.validation.type.bean.meaningfulgoals.CampaignWithMeaningfulGoalsIsMetrikaSourceOfValueBeforeApplyValidator.MeaningfulGoalWithIsMetrikaSourceOfValueValidator.validateIsMetrikaSourceOfValue;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.maxListSize;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.NumberConstraints.inRange;

@ParametersAreNonnullByDefault
public class CampaignWithMeaningfulGoalsValidator
        implements Validator<CampaignWithMeaningfulGoalsWithRequiredFields, Defect> {

    private final Set<Goal> availableCampaignGoals;
    private final CampaignValidationContainer campaignValidationContainer;
    private final boolean allGoalsAreAvailable;
    private final Set<String> enabledFeatures;

    public CampaignWithMeaningfulGoalsValidator(Set<Goal> availableCampaignGoals,
                                                CampaignValidationContainer campaignValidationContainer,
                                                boolean allGoalsAreAvailable,
                                                Set<String> enabledFeatures) {
        this.availableCampaignGoals = availableCampaignGoals;
        this.campaignValidationContainer = campaignValidationContainer;
        this.allGoalsAreAvailable = allGoalsAreAvailable;
        this.enabledFeatures = enabledFeatures;
    }

    @Override
    public ValidationResult<CampaignWithMeaningfulGoalsWithRequiredFields, Defect> apply(
            CampaignWithMeaningfulGoalsWithRequiredFields campaign) {
        var vb = ModelItemValidationBuilder.of(campaign);
        if (campaign.getCurrency() == null) {
            return vb.getResult();
        }
        MeaningfulGoalValidator validator = new MeaningfulGoalValidator(
                campaign,
                availableCampaignGoals,
                campaignValidationContainer,
                allGoalsAreAvailable, enabledFeatures);
        vb.list(CampaignWithMeaningfulGoalsWithRequiredFields.MEANINGFUL_GOALS)
                .checkEach(unique(MeaningfulGoal::getGoalId), When.notNull())
                .check(maxListSize(MEANINGFUL_GOALS_MAX_COUNT), When.notNull())
                .checkEachBy(validator, When.notNull());

        return vb.getResult();
    }


    public static class MeaningfulGoalValidator implements Validator<MeaningfulGoal, Defect> {
        private final CampaignWithMeaningfulGoalsWithRequiredFields campaign;
        private final Set<Long> availableCampaignGoalIds;
        private final Map<Long, Integer> counterIdByGoalId;
        private final Set<Integer> clientCounterIds;
        private final CampaignValidationContainer campaignValidationContainer;
        private final boolean allGoalsAreAvailable;
        private final Set<String> enabledFeatures;

        private MeaningfulGoalValidator(CampaignWithMeaningfulGoalsWithRequiredFields campaign,
                                        Set<Goal> availableCampaignGoals,
                                        CampaignValidationContainer campaignValidationContainer,
                                        boolean allGoalsAreAvailable, Set<String> enabledFeatures) {
            this.campaign = campaign;
            this.availableCampaignGoalIds = StreamEx.of(availableCampaignGoals)
                    .map(Goal::getId)
                    .append(ENGAGED_SESSION_GOAL_ID)
                    .toImmutableSet();
            this.counterIdByGoalId = StreamEx.of(availableCampaignGoals)
                    .mapToEntry(Goal::getId, Goal::getCounterId)
                    .nonNullValues()
                    .toMap();
            this.clientCounterIds = StreamEx.of(campaignValidationContainer.getMetrikaClient()
                            .getUsersCountersNumExtendedByCampaignCounterIds())
                    .flatCollection(UserCountersExtended::getCounters)
                    .map(CounterInfoDirect::getId)
                    .toSet();

            this.campaignValidationContainer = campaignValidationContainer;
            this.allGoalsAreAvailable = allGoalsAreAvailable;
            this.enabledFeatures = enabledFeatures;
        }

        @Override
        public ValidationResult<MeaningfulGoal, Defect> apply(MeaningfulGoal meaningfulGoal) {
            var vb = ModelItemValidationBuilder.of(meaningfulGoal);

            Currency currency = campaign.getCurrency().getCurrency();
            StrategyName strategyName = ifNotNull(campaign.getStrategy(), DbStrategyBase::getStrategyName);
            validateIsMetrikaSourceOfValue(vb, campaign.getSource(), strategyName,
                    counterIdByGoalId, clientCounterIds, enabledFeatures);
            vb.item(MeaningfulGoal.GOAL_ID)
                    .check(notNull())
                    .check(inSet(availableCampaignGoalIds),
                            When.isFalse(campaignValidationContainer.isCopy() || allGoalsAreAvailable));
            vb.item(MeaningfulGoal.CONVERSION_VALUE)
                    .check(notNull())
                    .check(inRange(currency.getMinPrice(), currency.getMaxAutobudget()));
            return vb.getResult();
        }
    }
}
