package ru.yandex.direct.core.entity.campaign.service.validation.type.bean;

import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.campaign.model.CampaignWithMetrikaCounters;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CollectionDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.metrikaCounterIsUnavailable;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.minListSize;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;

@ParametersAreNonnullByDefault
public class CampaignWithMetrikaCountersValidator implements Validator<CampaignWithMetrikaCounters, Defect> {

    private final Set<Long> availableCounterIds;
    private final Set<Long> systemCountersIds;
    private final boolean metrikaCountersAccessValidationOnSaveCampaignEnabled;
    private final int maxCountersCount;
    private final int minCountersCount;

    private CampaignWithMetrikaCountersValidator(Set<Long> availableCounterIds,
                                                 Set<Long> systemCountersIds,
                                                 Boolean metrikaCountersAccessValidationOnSaveCampaignEnabled,
                                                 int maxCountersCount,
                                                 int minCountersCount) {
        this.availableCounterIds = availableCounterIds;
        this.systemCountersIds = systemCountersIds;
        this.metrikaCountersAccessValidationOnSaveCampaignEnabled =
                metrikaCountersAccessValidationOnSaveCampaignEnabled;
        this.maxCountersCount = maxCountersCount;
        this.minCountersCount = minCountersCount;
    }

    public static CampaignWithMetrikaCountersValidator build(Set<Long> counterIdSet,
                                                             Set<Long> systemCountersIds,
                                                             boolean metrikaCountersAccessValidationOnSaveCampaignEnabled,
                                                             int maxCountersCount,
                                                             int minCountersCount) {
        return new CampaignWithMetrikaCountersValidator(counterIdSet, systemCountersIds,
                metrikaCountersAccessValidationOnSaveCampaignEnabled, maxCountersCount, minCountersCount);
    }

    @Override
    public ValidationResult<CampaignWithMetrikaCounters, Defect> apply(CampaignWithMetrikaCounters campaign) {
        var vb = ModelItemValidationBuilder.of(campaign);
        vb.list(CampaignWithMetrikaCounters.METRIKA_COUNTERS)
                .check(notNull(), When.isTrue(minCountersCount > 0));
        if (campaign.getMetrikaCounters() != null) {
            vb.list(CampaignWithMetrikaCounters.METRIKA_COUNTERS)
                    .check(minListSize(minCountersCount))
                    .check(maxClientMetrikaCountersCountOnCampaigns(
                            Math.min(maxCountersCount, campaign.getMaxSizeOfMetrikaCounters())))
                    .checkEach(notNull())
                    .checkEach(inSet(availableCounterIds), metrikaCounterIsUnavailable(),
                            When.isTrue(metrikaCountersAccessValidationOnSaveCampaignEnabled))
                    .checkEach(unique())
                    .checkEach(validId());
        }
        return vb.getResult();
    }

    private Constraint<List<Long>, Defect> maxClientMetrikaCountersCountOnCampaigns(int maxCountersCount) {
        return fromPredicate(counters -> StreamEx.of(counters)
                        .remove(systemCountersIds::contains)
                        .count() <= maxCountersCount,
                CollectionDefects.maxCollectionSize(maxCountersCount));
    }
}
