package ru.yandex.direct.core.entity.campaign.service.validation.type.bean;

import java.util.Set;

import ru.yandex.direct.core.entity.campaign.model.CampaignWithMobileContent;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;

public class CampaignWithMobileContentValidator implements Validator<CampaignWithMobileContent, Defect> {

    private Set<Long> clientMobileAppId;
    private final boolean allowNullDeviceTypeTargeting;
    private final boolean allowNullNetworkTargeting;
    private final boolean allowZeroMobileAppId;

    public CampaignWithMobileContentValidator(Set<Long> clientMobileAppId,
                                              boolean allowNullDeviceTypeTargeting,
                                              boolean allowNullNetworkTargeting,
                                              boolean allowZeroMobileAppId) {
        this.clientMobileAppId = clientMobileAppId;
        this.allowNullDeviceTypeTargeting = allowNullDeviceTypeTargeting;
        this.allowNullNetworkTargeting = allowNullNetworkTargeting;
        this.allowZeroMobileAppId = allowZeroMobileAppId;
    }

    public ValidationResult<CampaignWithMobileContent, Defect> apply(CampaignWithMobileContent campaign) {
        var vb = ModelItemValidationBuilder.of(campaign);

        vb.item(CampaignWithMobileContent.DEVICE_TYPE_TARGETING)
                .check(notNull(), When.isFalse(allowNullDeviceTypeTargeting))
                .check(notEmptyCollection(), When.isValid());
        vb.item(CampaignWithMobileContent.NETWORK_TARGETING)
                .check(notNull(), When.isFalse(allowNullNetworkTargeting))
                .check(notEmptyCollection(), When.isValid());
        vb.item(CampaignWithMobileContent.MOBILE_APP_ID)
                .check(notNull())
                .check(validMobileAppId(), When.notNull());
        return vb.getResult();
    }

    private Constraint<Long, Defect> validMobileAppId() {
        return fromPredicate(this::isValidMobileAppId, objectNotFound());
    }

    private boolean isValidMobileAppId(Long mobileAppId) {
        return (allowZeroMobileAppId && mobileAppId == 0L)
                || clientMobileAppId.contains(mobileAppId);
    }
}
