package ru.yandex.direct.core.entity.campaign.service.validation.type.bean;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.campaign.model.CampaignWithNetworkSettings;
import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform;
import ru.yandex.direct.core.entity.campaign.model.DbStrategyBase;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform.SEARCH;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MAX_CONTEXT_LIMIT;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MIN_CONTEXT_LIMIT;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.NO_CONTEXT_LIMIT;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.SHOWS_DISABLED_CONTEXT_LIMIT;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.defect.NumberDefects.inInterval;

@ParametersAreNonnullByDefault
public class CampaignWithNetworkSettingsValidator implements Validator<CampaignWithNetworkSettings, Defect> {

    private final RbacRole operatorRole;

    public CampaignWithNetworkSettingsValidator(RbacRole operatorRole) {
        this.operatorRole = operatorRole;
    }

    @Override
    public ValidationResult<CampaignWithNetworkSettings, Defect> apply(CampaignWithNetworkSettings campaign) {
        var vb = ModelItemValidationBuilder.of(campaign);
        var campaignsPlatform = ifNotNull(campaign.getStrategy(), DbStrategyBase::getPlatform);
        vb.item(CampaignWithNetworkSettings.CONTEXT_LIMIT)
                .check(notNull())
                .check(validContextLimit(campaignsPlatform), When.isValid());
        vb.item(CampaignWithNetworkSettings.ENABLE_CPC_HOLD)
                .check(notNull());
        return vb.getResult();
    }

    private Constraint<Integer, Defect> validContextLimit(CampaignsPlatform campaignsPlatform) {
        return fromPredicate(contextLimit -> isValidContextLimit(contextLimit, operatorRole,
                SEARCH.equals(campaignsPlatform)),
                inInterval(MIN_CONTEXT_LIMIT, MAX_CONTEXT_LIMIT));
    }

    public static boolean isValidContextLimit(Integer contextLimit,
                                              RbacRole operatorRole,
                                              boolean isSearchPlatform) {
        //0 не может быть входным параметром, но может быть проставлен самой операцией.
        return contextLimit == 0 || (contextLimit >= MIN_CONTEXT_LIMIT && contextLimit <= MAX_CONTEXT_LIMIT)
                || (RbacRole.SUPER == operatorRole && NO_CONTEXT_LIMIT.equals(contextLimit))
                || (isSearchPlatform && SHOWS_DISABLED_CONTEXT_LIMIT.equals(contextLimit));
    }
}
