package ru.yandex.direct.core.entity.campaign.service.validation.type.bean;

import java.util.Objects;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.campaign.model.CampaignWithPackageStrategy;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPricePackage;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils;
import ru.yandex.direct.core.entity.strategy.model.BaseStrategy;
import ru.yandex.direct.core.entity.strategy.utils.StrategyComparingUtils;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils.CAMPAIGN_TO_STRATEGY_TYPE;
import static ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.campaignStrategyInfoIsIgnoredOnCampaignAdd;
import static ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.inconsistentStrategyToCampaignType;
import static ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.strategyInfoMissed;
import static ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.strategyNotFound;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;

@ParametersAreNonnullByDefault
public class CampaignWithPackageStrategyAddPreValidator implements Validator<CampaignWithPackageStrategy, Defect> {

    private final BaseStrategy strategyWithCampaignStrategyId;
    private final BaseStrategy strategyFromCampaignStrategyWithCampaignStrategyId;

    public CampaignWithPackageStrategyAddPreValidator(@Nullable BaseStrategy strategyWithCampaignStrategyId,
                                                      @Nullable BaseStrategy strategyFromCampaignStrategyWithCampaignStrategyId) {
        this.strategyWithCampaignStrategyId = strategyWithCampaignStrategyId;
        this.strategyFromCampaignStrategyWithCampaignStrategyId = strategyFromCampaignStrategyWithCampaignStrategyId;
    }

    @Override
    public ValidationResult<CampaignWithPackageStrategy, Defect> apply(CampaignWithPackageStrategy campaignWithPackageStrategy) {
        var vb = ModelItemValidationBuilder.of(campaignWithPackageStrategy);

        vb.item(CampaignWithPackageStrategy.STRATEGY_ID)
                .check(validId(), When.notNull())
                .check(strategyIsAvailable(), When.isValidAnd(When.notNull()))
                .check(containsInfoAboutStrategy(campaignWithPackageStrategy))
                .check(consistentStrategyTypeToCampaignType(campaignWithPackageStrategy),
                        When.isValidAnd(When.notNull()))
                .weakCheck(consistentStrategyToPackageStrategy(), When.isValidAnd(When.notNull()));

        return vb.getResult();
    }

    private Constraint<Long, Defect> strategyIsAvailable() {
        return Constraint.fromPredicate(id -> Objects.nonNull(strategyWithCampaignStrategyId), strategyNotFound());
    }

    private Constraint<Long, Defect> containsInfoAboutStrategy(CampaignWithPackageStrategy campaignWithPackageStrategy) {
        return Constraint.fromPredicateOfNullable(
                strategyId -> {
                    DbStrategy dbStrategy = campaignWithPackageStrategy.getStrategy();
                    return dbStrategy != null || strategyId != null || campaignWithPackageStrategy instanceof CampaignWithPricePackage;
                },
                strategyInfoMissed()
        );
    }

    private Constraint<Long, Defect> consistentStrategyTypeToCampaignType(CampaignWithPackageStrategy campaign) {
        return Constraint.fromPredicate(
                strategyId -> {
                    var campaignStrategyName =
                            CampaignWithStrategyValidationUtils.toStrategyName(strategyWithCampaignStrategyId.getType());
                    return CAMPAIGN_TO_STRATEGY_TYPE.get(campaign.getType()).contains(campaignStrategyName);
                },
                inconsistentStrategyToCampaignType()
        );
    }

    private Constraint<Long, Defect> consistentStrategyToPackageStrategy() {
        return Constraint.fromPredicate(
                strategyId -> {
                    if (strategyFromCampaignStrategyWithCampaignStrategyId != null) {
                        return areEqualStrategies(strategyWithCampaignStrategyId,
                                strategyFromCampaignStrategyWithCampaignStrategyId);
                    }

                    return true;
                },
                campaignStrategyInfoIsIgnoredOnCampaignAdd()
        );
    }

    private boolean areEqualStrategies(BaseStrategy strategy1,
                                       BaseStrategy strategy2) {
        return !StrategyComparingUtils.areDifferentStrategies(strategy1, strategy2);
    }
}
