package ru.yandex.direct.core.entity.campaign.service.validation.type.bean;

import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithCampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPackageStrategy;
import ru.yandex.direct.core.entity.strategy.model.BaseStrategy;
import ru.yandex.direct.core.entity.strategy.model.CommonStrategy;
import ru.yandex.direct.core.entity.strategy.model.StrategyWithCampaignIds;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.campaignsWithDifferentTypesInOnePackage;
import static ru.yandex.direct.core.entity.campaign.service.validation.type.bean.utils.CampaignWithPackageStrategyValidatorUtils.hasWallet;
import static ru.yandex.direct.core.entity.campaign.service.validation.type.bean.utils.CampaignWithPackageStrategyValidatorUtils.isNonPublic;
import static ru.yandex.direct.core.entity.campaign.service.validation.type.bean.utils.CampaignWithPackageStrategyValidatorUtils.maxNumberOfCidsNotExceeded;
import static ru.yandex.direct.core.entity.campaign.service.validation.type.bean.utils.CampaignWithPackageStrategyValidatorUtils.strategyIsAvailableForPublication;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;

@ParametersAreNonnullByDefault
public class CampaignWithPackageStrategyAddValidator implements Validator<CampaignWithPackageStrategy, Defect> {

    private final BaseStrategy campaignStrategy;
    private final CampaignType campaignStrategyLinkedCampaignsType;
    private final List<CampaignWithPackageStrategy> fromOperationCampaigns;
    private final int maxNumberOfLinkedCids;

    public CampaignWithPackageStrategyAddValidator(@Nullable BaseStrategy campaignStrategy,
                                                   @Nullable CampaignType campaignStrategyLinkedCampaignsType,
                                                   List<CampaignWithPackageStrategy> fromOperationCampaigns,
                                                   int maxNumberOfLinkedCids) {
        this.campaignStrategy = campaignStrategy;
        this.campaignStrategyLinkedCampaignsType = campaignStrategyLinkedCampaignsType;
        this.fromOperationCampaigns = fromOperationCampaigns;
        this.maxNumberOfLinkedCids = maxNumberOfLinkedCids;
    }

    @Override
    public ValidationResult<CampaignWithPackageStrategy, Defect> apply(CampaignWithPackageStrategy campaign) {
        var vb = ModelItemValidationBuilder.of(campaign);
        if (campaignStrategy == null) {
            return vb.getResult();
        }

        List<CampaignWithPackageStrategy> newPotentialLinkedCampaigns =
                campaignsWithSameStrategyIdAfterApply(campaignStrategy.getId(), fromOperationCampaigns);

        var potentialNumberOfLinkedCampaigns = getPotentialNumberOfLinkedCids(newPotentialLinkedCampaigns);
        vb.item(CampaignWithPackageStrategy.STRATEGY_ID)
                .check(maxNumberOfCidsNotExceeded(potentialNumberOfLinkedCampaigns, maxNumberOfLinkedCids));
        vb.check(willContainCampaignsWithOneType(newPotentialLinkedCampaigns));

        if (strategyWillBecomePublicAfterOperation(newPotentialLinkedCampaigns)) {
            vb.check(hasWallet());
            vb.check(strategyIsAvailableForPublication(campaignStrategy.getType()));
        }

        return vb.getResult();
    }

    private List<CampaignWithPackageStrategy> campaignsWithSameStrategyIdAfterApply(Long strategyId,
                                                                                    List<CampaignWithPackageStrategy> fromOperationCampaigns) {
        return filterList(fromOperationCampaigns, c -> strategyId.equals(c.getStrategyId()));
    }

    private int getPotentialNumberOfLinkedCids(List<CampaignWithPackageStrategy> potentialLinkedCampaigns) {
        var strategyCids = ((StrategyWithCampaignIds) campaignStrategy).getCids();
        int strategyCidsNum = strategyCids != null ? strategyCids.size() : 0;
        return potentialLinkedCampaigns.size() + strategyCidsNum;
    }

    private Constraint<CampaignWithPackageStrategy, Defect> willContainCampaignsWithOneType(List<CampaignWithPackageStrategy> potentialLinkedCampaigns) {
        return Constraint.fromPredicate(
                campaign -> StreamEx.of(potentialLinkedCampaigns)
                        .map(CampaignWithCampaignType::getType)
                        .append(campaignStrategyLinkedCampaignsType)
                        .nonNull() // на случай, если сampaignStrategyLinkedCampaignsType=null(нет привязанных кампаний)
                        .distinct()
                        .count() <= 1,
                campaignsWithDifferentTypesInOnePackage());
    }

    private boolean strategyWillBecomePublicAfterOperation(List<CampaignWithPackageStrategy> newPotentialLinkedCampaigns) {
        return newPotentialLinkedCampaigns.size() != 0 && isNonPublic((CommonStrategy) campaignStrategy);
    }

}
