package ru.yandex.direct.core.entity.campaign.service.validation.type.bean;

import java.util.Objects;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.campaign.model.CampaignWithPackageStrategy;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPricePackage;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStrategy;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils;
import ru.yandex.direct.core.entity.strategy.model.BaseStrategy;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelChangesValidationBuilder;

import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils.CAMPAIGN_TO_STRATEGY_TYPE;
import static ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.campaignNotInPackage;
import static ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.campaignStrategyInfoIsIgnoredOnCampaignUpdate;
import static ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.inconsistentStrategyToCampaignType;
import static ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.newStrategyHasBeenCreated;
import static ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.strategyInfoMissed;
import static ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.strategyNotFound;
import static ru.yandex.direct.core.entity.campaign.service.validation.type.bean.CampaignWithPackageStrategyUpdatePreValidatorHelper.isModelChangesAffectsStrategy;
import static ru.yandex.direct.core.entity.campaign.service.validation.type.bean.CampaignWithPackageStrategyUpdatePreValidatorHelper.isNotModelChangesAffectsStrategy;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;

@ParametersAreNonnullByDefault
public class CampaignWithPackageStrategyUpdatePreValidator implements Validator<ModelChanges<CampaignWithPackageStrategy>, Defect> {
    private final BaseStrategy packageStrategy;
    private final CampaignWithPackageStrategy campaignWithPackageStrategy;
    private final boolean isCampaignWithPublicStrategy;

    public CampaignWithPackageStrategyUpdatePreValidator(@Nullable BaseStrategy packageStrategy,
                                                         CampaignWithPackageStrategy campaignWithPackageStrategy,
                                                         boolean isCampaignWithPublicStrategy) {
        this.packageStrategy = packageStrategy;
        this.campaignWithPackageStrategy = campaignWithPackageStrategy;
        this.isCampaignWithPublicStrategy = isCampaignWithPublicStrategy;
    }

    @Override
    public ValidationResult<ModelChanges<CampaignWithPackageStrategy>, Defect> apply(ModelChanges<CampaignWithPackageStrategy> modelChanges) {
        ModelChangesValidationBuilder<CampaignWithPackageStrategy> vb =
                ModelChangesValidationBuilder.of(modelChanges);

        var isStrategyPropChanged = modelChanges.isPropChanged(CampaignWithPackageStrategy.STRATEGY);
        var strategyIdActuallyChanges =
                modelChanges.isPropChanged(CampaignWithPackageStrategy.STRATEGY_ID) &&
                        !Objects.equals(
                                modelChanges.getPropIfChanged(CampaignWithPackageStrategy.STRATEGY_ID),
                                campaignWithPackageStrategy.getStrategyId()
                        );
        var forceUnbind = strategyIdActuallyChanges &&
                modelChanges.getChangedProp(CampaignWithPackageStrategy.STRATEGY_ID) == null;

        vb.item(CampaignWithPackageStrategy.STRATEGY_ID)
                .check(validId(), When.notNull())
                .check(strategyIsAvailable(), When.isValidAnd(When.notNull()))
                .check(containsInfoAboutStrategy(modelChanges), When.isTrue(isStrategyPropChanged))
                .check(consistentPackageStrategyToCampaignType(), When.isValidAnd(When.notNull()));

        vb.check(nonPublicStrategyNotReset(), When.isValid());

        vb.weakCheck(changingStrategyIsEqualToNewPackageStrategy(),
                When.isValidAnd(When.isTrue(strategyIdActuallyChanges && !forceUnbind)));
        if (packageStrategy != null) {
            vb.weakCheck(
                    campaignIsRemovedFromPackageWhenPublicStrategyChanging(packageStrategy),
                    When.isValidAnd(When.isTrue(!strategyIdActuallyChanges && isCampaignWithPublicStrategy)));
        }

        return vb.getResult();
    }

    private Constraint<Long, Defect> strategyIsAvailable() {
        return Constraint.fromPredicate(strategyId -> Objects.nonNull(packageStrategy), strategyNotFound());
    }

    private Constraint<Long, Defect> containsInfoAboutStrategy(ModelChanges<CampaignWithPackageStrategy> modelChanges) {
        return Constraint.fromPredicateOfNullable(
                strategyId -> {
                    DbStrategy dbStrategy = modelChanges.getChangedProp(CampaignWithStrategy.STRATEGY);
                    return dbStrategy != null || strategyId != null || CampaignWithPricePackage.class.isAssignableFrom(modelChanges.getModelType());
                },
                strategyInfoMissed()
        );
    }

    private Constraint<Long, Defect> consistentPackageStrategyToCampaignType() {
        return Constraint.fromPredicate(
                strategyId -> {
                    var campaignStrategyName = packageStrategy != null ?
                            CampaignWithStrategyValidationUtils.toStrategyName(packageStrategy.getType()) : null;

                    return campaignStrategyName == null ||
                            CAMPAIGN_TO_STRATEGY_TYPE.get(campaignWithPackageStrategy.getType()).contains(campaignStrategyName);
                },
                inconsistentStrategyToCampaignType()
        );
    }

    private Constraint<ModelChanges<CampaignWithPackageStrategy>, Defect> nonPublicStrategyNotReset() {
        return Constraint.fromPredicateOfNullable(
                mc -> !mc.isPropChanged(CampaignWithPackageStrategy.STRATEGY_ID) ||
                        mc.getChangedProp(CampaignWithPackageStrategy.STRATEGY_ID) != null ||
                        isCampaignWithPublicStrategy,
                campaignNotInPackage()
        );
    }

    private Constraint<ModelChanges<CampaignWithPackageStrategy>, Defect> changingStrategyIsEqualToNewPackageStrategy() {
        return Constraint.fromPredicate(
                changes -> !isModelChangesAffectsStrategy(changes),
                campaignStrategyInfoIsIgnoredOnCampaignUpdate()
        );
    }

    private Constraint<ModelChanges<CampaignWithPackageStrategy>, Defect> campaignIsRemovedFromPackageWhenPublicStrategyChanging(BaseStrategy strategy) {
        return Constraint.fromPredicate(
                changes -> isNotModelChangesAffectsStrategy(changes, strategy),
                newStrategyHasBeenCreated()
        );
    }
}
