package ru.yandex.direct.core.entity.campaign.service.validation.type.bean;

import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithCampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPackageStrategy;
import ru.yandex.direct.core.entity.strategy.model.BaseStrategy;
import ru.yandex.direct.core.entity.strategy.model.CommonStrategy;
import ru.yandex.direct.core.entity.strategy.model.StrategyWithCampaignIds;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.campaignsWithDifferentTypesInOnePackage;
import static ru.yandex.direct.core.entity.campaign.service.validation.type.bean.utils.CampaignWithPackageStrategyValidatorUtils.hasWallet;
import static ru.yandex.direct.core.entity.campaign.service.validation.type.bean.utils.CampaignWithPackageStrategyValidatorUtils.isNonPublic;
import static ru.yandex.direct.core.entity.campaign.service.validation.type.bean.utils.CampaignWithPackageStrategyValidatorUtils.maxNumberOfCidsNotExceeded;
import static ru.yandex.direct.core.entity.campaign.service.validation.type.bean.utils.CampaignWithPackageStrategyValidatorUtils.strategyIsAvailableForPublication;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public class CampaignWithPackageStrategyUpdateValidator implements Validator<CampaignWithPackageStrategy, Defect> {

    private final BaseStrategy campaignStrategy;
    private final CampaignType linkedCampaignsType;
    private final List<CampaignWithPackageStrategy> fromOperationCampaigns;
    private final int maxNumberOfLinkedCids;

    public CampaignWithPackageStrategyUpdateValidator(@Nullable BaseStrategy newCampaignStrategy,
                                                      @Nullable CampaignType linkedCampaignsType,
                                                      List<CampaignWithPackageStrategy> fromOperationCampaigns,
                                                      int maxNumberOfLinkedCids) {
        this.campaignStrategy = newCampaignStrategy;
        this.linkedCampaignsType = linkedCampaignsType;
        this.fromOperationCampaigns = fromOperationCampaigns;
        this.maxNumberOfLinkedCids = maxNumberOfLinkedCids;
    }

    @Override
    public ValidationResult<CampaignWithPackageStrategy, Defect> apply(CampaignWithPackageStrategy campaign) {
        var vb = ModelItemValidationBuilder.of(campaign);
        if (campaignStrategy == null) {
            return vb.getResult();
        }

        var allPotentialLinkedCampaignIds = campaignsWithSameStrategyIdAfterApplyIds(campaignStrategy.getId(),
                ((StrategyWithCampaignIds) campaignStrategy).getCids(), fromOperationCampaigns);
        vb.item(CampaignWithPackageStrategy.STRATEGY_ID)
                .check(maxNumberOfCidsNotExceeded(allPotentialLinkedCampaignIds.size(), maxNumberOfLinkedCids))
                .check(willContainCampaignsWithOneType(allPotentialLinkedCampaignIds));

        if (strategyWillBecomePublicAfterOperation(allPotentialLinkedCampaignIds)) {
            vb.check(hasWallet());
            vb.check(strategyIsAvailableForPublication(campaignStrategy.getType()));
        }

        return vb.getResult();
    }

    private Set<Long> campaignsWithSameStrategyIdAfterApplyIds(Long strategyId, @Nullable List<Long> strategyCids,
                                                               List<CampaignWithPackageStrategy> campaigns) {
        Set<Long> oldStrategyCids = strategyCids == null ? Set.of() : Set.copyOf(strategyCids);
        List<Long> cidsFromOperation = mapList(campaigns, CampaignWithPackageStrategy::getId);

        Set<Long> campaignWithSameStrategyIdAfterApplyIds = campaigns.stream()
                .filter(c -> strategyId.equals(c.getStrategyId()))
                .map(BaseCampaign::getId)
                .collect(Collectors.toSet());

        return StreamEx.of(oldStrategyCids)
                .filter(cid -> !cidsFromOperation.contains(cid))
                .append(campaignWithSameStrategyIdAfterApplyIds)
                .toSet();
    }

    private Constraint<Long, Defect> willContainCampaignsWithOneType(Set<Long> potentialLinkedCampaignsIds) {
        return Constraint.fromPredicate(
                id -> StreamEx.of(fromOperationCampaigns)
                        .mapToEntry(CampaignWithPackageStrategy::getId, CampaignWithCampaignType::getType)
                        .filterKeys(potentialLinkedCampaignsIds::contains)
                        .values()
                        .append(linkedCampaignsType)
                        .nonNull()  // на случай, если linkedCampaignsType=null(нет привязанных кампаний)
                        .distinct()
                        .count() <= 1,
                campaignsWithDifferentTypesInOnePackage());
    }

    private boolean strategyWillBecomePublicAfterOperation(Set<Long> allPotentialLinkedCampaignIds) {
        List<Long> strategyCids = ((StrategyWithCampaignIds) campaignStrategy).getCids();
        Set<Long> oldStrategyCids = strategyCids == null
                ? Set.of()
                : Set.copyOf(strategyCids);
        return !oldStrategyCids.containsAll(allPotentialLinkedCampaignIds) && isNonPublic((CommonStrategy) campaignStrategy);
    }

}
