package ru.yandex.direct.core.entity.campaign.service.validation.type.bean;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.collections4.CollectionUtils;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPricePackage;
import ru.yandex.direct.core.entity.campaign.model.PriceFlightTargetingsSnapshot;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageCampaignOptions;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageWithoutClients;
import ru.yandex.direct.core.entity.pricepackage.model.ShowsFrequencyLimit;
import ru.yandex.direct.core.entity.pricepackage.model.TargetingsCustom;
import ru.yandex.direct.core.entity.pricepackage.model.TargetingsFixed;
import ru.yandex.direct.core.entity.pricepackage.model.ViewType;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.constraint.CommonConstraints;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.DefaultValidator;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static com.google.common.base.Preconditions.checkNotNull;
import static org.apache.commons.collections4.CollectionUtils.isNotEmpty;
import static ru.yandex.direct.core.entity.campaign.service.CampaignWithPricePackageUtils.extractImpressionRateIntervalDaysFromPackage;
import static ru.yandex.direct.core.entity.campaign.service.CampaignWithPricePackageUtils.pricePackageAllowsImpressionRate;
import static ru.yandex.direct.core.entity.campaign.service.CampaignWithPricePackageUtils.pricePackageHasImpressionRate;
import static ru.yandex.direct.core.entity.pricepackage.service.validation.PricePackageValidator.REGION_TYPE_REGION;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.isEmptyCollection;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isEqual;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.DateConstraints.isNotAfterThan;
import static ru.yandex.direct.validation.constraint.DateConstraints.isNotBeforeThan;
import static ru.yandex.direct.validation.constraint.NumberConstraints.inRange;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;
import static ru.yandex.direct.validation.defect.CommonDefects.mustBeEmpty;

@ParametersAreNonnullByDefault
public class CampaignWithPricePackageValidator<T extends PricePackageWithoutClients> implements
        DefaultValidator<CampaignWithPricePackage> {
    private final Map<Long, T> availablePricePackages;
    private final Set<String> availableFeatures;

    public CampaignWithPricePackageValidator(Map<Long, T> availablePricePackages, Set<String> availableFeatures) {
        this.availablePricePackages = availablePricePackages;
        this.availableFeatures = availableFeatures;
    }

    @Override
    public ValidationResult<CampaignWithPricePackage, Defect> apply(CampaignWithPricePackage campaign) {
        var pricePackage = availablePricePackages.get(campaign.getPricePackageId());
        checkNotNull(pricePackage);

        ModelItemValidationBuilder<CampaignWithPricePackage> vb = ModelItemValidationBuilder.of(campaign);

        vb.item(CampaignWithPricePackage.FLIGHT_ORDER_VOLUME)
                .check(notNull())
                .check(inRange(pricePackage.getOrderVolumeMin(), pricePackage.getOrderVolumeMax()), When.isValid());

        vb.item(CampaignWithPricePackage.FLIGHT_TARGETINGS_SNAPSHOT)
                .check(notNull())
                .checkBy(
                        targetingsSnapshot -> checkTargetings(targetingsSnapshot, pricePackage, this.availableFeatures),
                        When.isValid()
                );

        vb.item(CampaignWithPricePackage.START_DATE)
                .check(notNull())
                .check(isNotBeforeThan(pricePackage.getDateStart()), When.isValid());

        vb.item(CampaignWithPricePackage.END_DATE)
                .check(notNull())
                .check(isNotAfterThan(pricePackage.getDateEnd()), When.isValid());

        vb.item(CampaignWithPricePackage.CURRENCY)
                .check(notNull())
                .check(fromPredicate(campaignCurrency -> pricePackage.getCurrency().equals(campaignCurrency),
                        invalidValue()), When.isValid());

        vb.list(CampaignWithPricePackage.BRAND_SAFETY_CATEGORIES)
                .check(isEmptyCollection(), mustBeEmpty(),
                        When.isValidAnd(When.isFalse(pricePackageAllowsBrandSafetyCategories(pricePackage))));

        vb.list(CampaignWithPricePackage.DISABLED_VIDEO_PLACEMENTS)
                .check(isEmptyCollection(), mustBeEmpty(),
                        When.isValidAnd(When.isFalse(pricePackageAllowsDisabledVideoPlacements(pricePackage))));

        vb.list(CampaignWithPricePackage.DISABLED_DOMAINS)
                .check(isEmptyCollection(), mustBeEmpty(),
                        When.isValidAnd(When.isFalse(pricePackageAllowsDisabledPlaces(pricePackage))));

        vb.checkBy(camp -> validateImpressionRateLimit(camp, pricePackage), When.isValid());

        return vb.getResult();
    }

    private static boolean pricePackageAllowsBrandSafetyCategories(PricePackageWithoutClients pricePackage) {
        return Optional.of(pricePackage)
                .map(PricePackageWithoutClients::getCampaignOptions)
                .map(PricePackageCampaignOptions::getAllowBrandSafety)
                .filter(flag -> !flag)
                .isEmpty();
    }

    private static boolean pricePackageAllowsDisabledVideoPlacements(PricePackageWithoutClients pricePackage) {
        return Optional.of(pricePackage)
                .map(PricePackageWithoutClients::getCampaignOptions)
                .map(PricePackageCampaignOptions::getAllowDisabledVideoPlaces)
                .filter(flag -> !flag)
                .isEmpty();
    }

    private static boolean pricePackageAllowsDisabledPlaces(PricePackageWithoutClients pricePackage) {
        return Optional.of(pricePackage)
                .map(PricePackageWithoutClients::getCampaignOptions)
                .map(PricePackageCampaignOptions::getAllowDisabledPlaces)
                .filter(flag -> !flag)
                .isEmpty();
    }

    private static ValidationResult<PriceFlightTargetingsSnapshot, Defect> checkTargetings(
            PriceFlightTargetingsSnapshot snapshot, PricePackageWithoutClients pricePackage,
            Set<String> availableFeatures
    ) {
        TargetingsCustom custom = pricePackage.getTargetingsCustom();
        TargetingsFixed fixed = pricePackage.getTargetingsFixed();
        Integer packageGeoType = nvl(fixed.getGeoType(), custom.getGeoType());
        boolean geoType10 = packageGeoType.equals(REGION_TYPE_REGION);//удалим в DIRECT-137986
        boolean cpmYndxFrontpagePackage = pricePackage.getAvailableAdGroupTypes() == null
                || pricePackage.getAvailableAdGroupTypes().contains(AdGroupType.CPM_YNDX_FRONTPAGE);

        var vb = ModelItemValidationBuilder.of(snapshot);
        vb.item(PriceFlightTargetingsSnapshot.GEO_TYPE)
                .check(notNull(), When.isFalse(geoType10))
                .check(checkGeoType(fixed.getGeoType(), custom.getGeoType()), When.isValidAnd(When.isFalse(geoType10)));
        vb.item(PriceFlightTargetingsSnapshot.GEO_EXPANDED)
                .check(notNull())
                .check(notEmptyCollection(), When.isValid())
                .check(checkGeoExpanded(fixed.getGeoExpanded(), custom.getGeoExpanded()), When.isValidAnd(
                        When.isTrue(nvl(packageGeoType, REGION_TYPE_REGION) != REGION_TYPE_REGION)));
        //Специфичные поля для CPM_YNDX_FRONTPAGE
        if (cpmYndxFrontpagePackage) {
            vb.item(PriceFlightTargetingsSnapshot.VIEW_TYPES)
                    .check(notNull())
                    .check(checkViewTypes(fixed.getViewTypes()));
            vb.item(PriceFlightTargetingsSnapshot.ALLOW_EXPANDED_DESKTOP_CREATIVE)
                    .check(notNull())
                    .check(isEqual(fixed.getAllowExpandedDesktopCreative(), invalidValue()), When.isValid());
        }

        return vb.getResult();
    }

    private static Constraint<Integer, Defect> checkGeoType(@Nullable Integer fixedGeoType,
                                                            @Nullable Integer customGeoType) {
        return fromPredicate(snapshotGeoType -> snapshotGeoType.equals(nvl(fixedGeoType, customGeoType)),
                invalidValue());
    }

    private static Constraint<List<Long>, Defect> checkGeoExpanded(List<Long> fixedGeoExpanded,
                                                                   List<Long> customGeoExpanded) {
        return fromPredicate(snapshotGeoExpanded -> {
            if (isNotEmpty(fixedGeoExpanded)) {
                // сравниваем игнорируя порядок
                return CollectionUtils.isEqualCollection(fixedGeoExpanded, snapshotGeoExpanded);
            } else {
                return customGeoExpanded.containsAll(snapshotGeoExpanded);
            }
        }, invalidValue());
    }

    private static Constraint<List<ViewType>, Defect> checkViewTypes(List<ViewType> fixedViewTypes) {
        return CommonConstraints.isEqualCollection(fixedViewTypes, invalidValue());
    }

    /**
     * Проверка, что параметры ограничения частоты показов разрешены пакетом.
     * <p>
     * Если пакет запрещает выставлять ограничение, оба параметра должны быть {@code null}.
     * <p>
     * Если в пакете выставлено фиксированное ограничение, должны быть переданы либо {@code null}, либо такие же,
     * как на пакете
     */
    private ValidationResult<CampaignWithPricePackage, Defect> validateImpressionRateLimit(
            CampaignWithPricePackage campaign,
            PricePackageWithoutClients pricePackage) {
        var vb = ModelItemValidationBuilder.of(campaign);
        ShowsFrequencyLimit limit = Optional.ofNullable(pricePackage)
                .map(PricePackageWithoutClients::getCampaignOptions)
                .map(PricePackageCampaignOptions::getShowsFrequencyLimit)
                .orElse(null);
        boolean minLimit = Optional.ofNullable(limit).map(ShowsFrequencyLimit::getMinLimit).orElse(false);
        vb.item(CampaignWithPricePackage.IMPRESSION_RATE_COUNT)
                .check(isNull(), When.isFalse(pricePackageAllowsImpressionRate(pricePackage)))
                .check(notNull(), When.isTrue(minLimit))
                // используем fromPredicate вместо fromPredicateOfNullable,
                // т.к. разрешаем передавать null, если параметр будет скопирован из пакета
                .check(fromPredicate(rateCount -> limit.getFrequencyLimit().equals(rateCount), invalidValue()),
                        When.isTrue(pricePackageHasImpressionRate(pricePackage)))
                .check(fromPredicate(rateCount -> limit.getFrequencyLimit() <= rateCount, invalidValue()),
                        When.isTrue(minLimit));
        vb.item(CampaignWithPricePackage.IMPRESSION_RATE_INTERVAL_DAYS)
                .check(isNull(), When.isFalse(pricePackageAllowsImpressionRate(pricePackage)))
                .check(notNull(), When.isTrue(minLimit))
                // используем fromPredicate вместо fromPredicateOfNullable,
                //  т.к. разрешаем не указывать параметр, если он будет скопирован из пакета
                .check(fromPredicate(
                        days -> Objects.equals(extractImpressionRateIntervalDaysFromPackage(pricePackage), days),
                        invalidValue()),
                        When.isTrue(pricePackageHasImpressionRate(pricePackage)))
                .check(fromPredicate(days -> limit.getFrequencyLimitDays() <= days, invalidValue()),
                        When.isTrue(minLimit));
        return vb.getResult();
    }
}
