package ru.yandex.direct.core.entity.campaign.service.validation.type.bean;

import java.time.LocalDate;
import java.util.Arrays;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.common.net.NetAcl;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.service.validation.type.CommonCampaignBeanValidatorContext;
import ru.yandex.direct.core.entity.campaign.service.validation.type.TimeIntervalValidator;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MAX_CAMPAIGN_FIO_LENGTH;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MAX_CAMPAIGN_NAME_LENGTH;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MAX_CAMPAIGN_WARNING_BALANCE;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MAX_DISABLED_IPS_COUNT;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MIN_CAMPAIGN_WARNING_BALANCE;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstraints.disabledIpIsNotInternal;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstraints.disabledIpIsNotPrivate;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstraints.disabledIpIsValid;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.maxListSize;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.DateConstraints.endDateIsNotBeforeThan;
import static ru.yandex.direct.validation.constraint.NumberConstraints.inRange;
import static ru.yandex.direct.validation.constraint.StringConstraints.hasNoForbiddenChars;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.constraint.StringConstraints.notControlChars_AdmissibleChars;
import static ru.yandex.direct.validation.constraint.StringConstraints.notEmpty2;
import static ru.yandex.direct.validation.constraint.StringConstraints.onlyUtf8Mb3Symbols;
import static ru.yandex.direct.validation.constraint.StringConstraints.validEmailInvalidFormat;

@ParametersAreNonnullByDefault
public class CommonCampaignBeanValidator implements Validator<CommonCampaign, Defect> {

    private static final List<String> FORBIDDEN_CHARS_IN_FIO = Arrays.asList("<", ">");

    private static final List<String> FORBIDDEN_CHARS_IN_CAMPAIGN_NAME = Arrays.asList("<", ">");

    private final CommonCampaignBeanValidatorContext context;

    // В базе есть данные с FIO длинее 255
    // dbs-sql pr:ppc:all "select count(*) from camp_options where length(FIO) > 255 \G"
    // на всякий случай валидируем длину fio не всегда
    private final boolean limitFioTo255Chars;

    // В базе много странных FIO, поэтому валидируем не всегда
    // dbs-sql pr:ppc:all "select count(*) from camp_options where FIO like '%<%' OR FIO LIKE '%>%' \G"
    private final boolean validateFioForForbiddenChars;

    // интерфейс вообще разрешает символы "<>", поэтому делаем только для апи5
    private final boolean validateCampaignNameForForbiddenChars;

    private CommonCampaignBeanValidator(CommonCampaignBeanValidatorContext context,
                                        boolean limitFioTo255Chars,
                                        boolean validateFioForForbiddenChars,
                                        boolean validateCampaignNameForForbiddenChars) {
        this.context = context;
        this.limitFioTo255Chars = limitFioTo255Chars;
        this.validateFioForForbiddenChars = validateFioForForbiddenChars;
        this.validateCampaignNameForForbiddenChars = validateCampaignNameForForbiddenChars;
    }

    public static CommonCampaignBeanValidator build(CommonCampaignBeanValidatorContext context,
                                                    boolean limitFioTo255Chars,
                                                    boolean validateFioForForbiddenChars,
                                                    boolean validateCampaignNameForForbiddenChars) {
        return new CommonCampaignBeanValidator(context, limitFioTo255Chars, validateFioForForbiddenChars,
                validateCampaignNameForForbiddenChars);
    }

    @Override
    public ValidationResult<CommonCampaign, Defect> apply(CommonCampaign commonCampaign) {
        var vb = ModelItemValidationBuilder.of(commonCampaign);

        vb.item(CommonCampaign.NAME)
                .checkBy(this::validateName);

        vb.item(CommonCampaign.START_DATE)
                .checkBy(CommonCampaignBeanValidator::validateStartDate);

        vb.item(CommonCampaign.END_DATE)
                .checkBy(endDate -> validateEndDate(endDate, commonCampaign.getStartDate()));

        vb.item(CommonCampaign.DISABLED_IPS)
                .checkBy(x -> validateDisabledIps(x, context.getNetworks()));

        vb.item(CommonCampaign.SMS_TIME)
                .check(notNull())
                .checkBy(new TimeIntervalValidator(), When.notNull());
        vb.item(CommonCampaign.EMAIL)
                .check(notNull())
                .check(notEmpty2())
                .check(validEmailInvalidFormat(), When.isValid());
        vb.item(CommonCampaign.WARNING_BALANCE)
                .check(inRange(MIN_CAMPAIGN_WARNING_BALANCE, MAX_CAMPAIGN_WARNING_BALANCE));
        vb.item(CommonCampaign.ENABLE_PAUSED_BY_DAY_BUDGET_EVENT)
                .check(notNull());
        vb.item(CommonCampaign.IS_SERVICE_REQUESTED)
                .check(notNull());
        vb.item(CommonCampaign.FIO)
                .check(hasNoForbiddenChars(FORBIDDEN_CHARS_IN_FIO), When.isTrue(validateFioForForbiddenChars))
                .check(notControlChars_AdmissibleChars(), When.isTrue(validateFioForForbiddenChars))
                .check(maxStringLength(MAX_CAMPAIGN_FIO_LENGTH), When.isTrue(limitFioTo255Chars));

        return vb.getResult();
    }

    ValidationResult<String, Defect> validateName(String campaignName) {
        var vb = ItemValidationBuilder.of(campaignName, Defect.class);

        vb.check(notNull())
                .check(notBlank(), When.isValid())
                .check(onlyUtf8Mb3Symbols(), When.isValid())
                .check(hasNoForbiddenChars(FORBIDDEN_CHARS_IN_CAMPAIGN_NAME),
                        When.isTrue(validateCampaignNameForForbiddenChars))
                .check(maxStringLength(MAX_CAMPAIGN_NAME_LENGTH));

        return vb.getResult();
    }

    public static ValidationResult<LocalDate, Defect> validateStartDate(LocalDate startDate) {
        var vb = ItemValidationBuilder.of(startDate, Defect.class);

        vb.check(notNull());

        return vb.getResult();
    }

    static ValidationResult<LocalDate, Defect> validateEndDate(LocalDate endDate, @Nullable LocalDate startDate) {
        var vb = ItemValidationBuilder.of(endDate, Defect.class);
        if (startDate == null) {
            return vb.getResult();
        }

        vb.check(endDateIsNotBeforeThan(startDate), When.notNull());

        return vb.getResult();
    }

    static ValidationResult<List<String>, Defect> validateDisabledIps(List<String> disabledIps, NetAcl networks) {
        ListValidationBuilder<String, Defect> vb = ListValidationBuilder.of(disabledIps);

        vb.check(notEmptyCollection(), When.notNull())
                .check(maxListSize(MAX_DISABLED_IPS_COUNT))
                .checkEach(notNull())
                .checkEach(disabledIpIsValid(), When.isValid())
                .checkEach(disabledIpIsNotPrivate(), When.isValid())
                .checkEach(disabledIpIsNotInternal(networks), When.isValid());

        return vb.getResult();
    }
}
