package ru.yandex.direct.core.entity.campaign.service.validation.type.bean.meaningfulgoals;

import java.math.BigDecimal;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.campaign.model.CampaignWithMeaningfulGoalsWithRequiredFields;
import ru.yandex.direct.core.entity.campaign.model.MeaningfulGoal;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.campaign.model.CampaignWithMeaningfulGoals.MEANINGFUL_GOALS;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.ENGAGED_SESSION_GOAL_ID;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MEANINGFUL_GOALS_MAX_COUNT;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.maxListSize;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;


public class CampaignWithMeaningfulGoalsGoalIdBeforeApplyValidator
        implements Validator<ModelChanges<CampaignWithMeaningfulGoalsWithRequiredFields>, Defect> {

    private final Set<Goal> availableCampaignGoals;
    private final boolean allGoalsAreAvailable;
    private final CampaignWithMeaningfulGoalsWithRequiredFields currentCampaign;

    private CampaignWithMeaningfulGoalsGoalIdBeforeApplyValidator(
            @Nullable Set<Goal> availableCampaignGoals,
            boolean allGoalsAreAvailable,
            CampaignWithMeaningfulGoalsWithRequiredFields currentCampaign) {
        this.availableCampaignGoals = availableCampaignGoals;
        this.allGoalsAreAvailable = allGoalsAreAvailable;
        this.currentCampaign = currentCampaign;
    }

    public static CampaignWithMeaningfulGoalsGoalIdBeforeApplyValidator build(
            @Nullable Set<Goal> availableCampaignGoals,
            boolean allGoalsAreAvailable,
            CampaignWithMeaningfulGoalsWithRequiredFields currentCampaign) {
        return new CampaignWithMeaningfulGoalsGoalIdBeforeApplyValidator(availableCampaignGoals, allGoalsAreAvailable, currentCampaign);
    }


    @Override
    public ValidationResult<ModelChanges<CampaignWithMeaningfulGoalsWithRequiredFields>, Defect> apply(
            ModelChanges<CampaignWithMeaningfulGoalsWithRequiredFields> campaignChanges) {
        var vb = ItemValidationBuilder.of(campaignChanges, Defect.class);

        if (CampaignWithMeaningfulGoalsConversionValueBeforeApplyValidator.getCurrencyCode(campaignChanges, currentCampaign) == null) {
            return vb.getResult();
        }

        Map<Long, BigDecimal> currentConversationValueByGoalId = nvl(listToMap(currentCampaign.getMeaningfulGoals(),
                MeaningfulGoal::getGoalId, MeaningfulGoal::getConversionValue), Map.of());

        Set<Long> availableCampaignGoalIds = availableCampaignGoals == null ?
                Set.of(ENGAGED_SESSION_GOAL_ID) :
                StreamEx.of(availableCampaignGoals)
                        .map(Goal::getId)
                        .append(ENGAGED_SESSION_GOAL_ID)
                        .toImmutableSet();

        vb.list(campaignChanges.getPropIfChanged(MEANINGFUL_GOALS),
                MEANINGFUL_GOALS.name())
                .check(maxListSize(MEANINGFUL_GOALS_MAX_COUNT), When.notNull())
                .checkEach(unique(MeaningfulGoal::getGoalId), When.notNull())
                .checkEachBy(goal -> MeaningfulGoalWithGoalIdValidator.build(
                        availableCampaignGoalIds, currentConversationValueByGoalId, allGoalsAreAvailable).apply(goal),
                        When.notNull());
        return vb.getResult();
    }


    public static class MeaningfulGoalWithGoalIdValidator implements Validator<MeaningfulGoal, Defect> {
        private final Set<Long> availableCampaignGoalIds;
        private final Map<Long, BigDecimal> currentConversationValueByGoalId;
        private final boolean allGoalsAreAvailable;

        private MeaningfulGoalWithGoalIdValidator(Set<Long> availableCampaignGoalIds,
                                                  Map<Long, BigDecimal> currentConversationValueByGoalId,
                                                  boolean allGoalsAreAvailable) {
            this.availableCampaignGoalIds = availableCampaignGoalIds;
            this.currentConversationValueByGoalId = currentConversationValueByGoalId;
            this.allGoalsAreAvailable = allGoalsAreAvailable;
        }

        public static MeaningfulGoalWithGoalIdValidator build(Set<Long> availableCampaignGoalIds,
                                                              Map<Long, BigDecimal> currentConversionValueByGoalId,
                                                              boolean allGoalsAreAvailable) {
            return new MeaningfulGoalWithGoalIdValidator(availableCampaignGoalIds, currentConversionValueByGoalId, allGoalsAreAvailable);
        }

        @Override
        public ValidationResult<MeaningfulGoal, Defect> apply(MeaningfulGoal meaningfulGoal) {
            var vb = ModelItemValidationBuilder.of(meaningfulGoal);
            vb.item(MeaningfulGoal.GOAL_ID)
                    .check(notNull())
                    .check(inSet(availableCampaignGoalIds),
                            When.isFalse(currentConversationValueByGoalId.containsKey(meaningfulGoal.getGoalId())
                                    || allGoalsAreAvailable));
            return vb.getResult();
        }
    }
}
