package ru.yandex.direct.core.entity.campaign.service.validation.type.bean.meaningfulgoals;

import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.campaign.model.CampaignSource;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMeaningfulGoalsWithRequiredFields;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.DbStrategyBase;
import ru.yandex.direct.core.entity.campaign.model.MeaningfulGoal;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.campaign.service.CampaignStrategyUtils.isUnavailableGoalsAllowed;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MEANINGFUL_GOALS_MAX_COUNT;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nullableNvl;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.maxListSize;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notTrue;

public class CampaignWithMeaningfulGoalsIsMetrikaSourceOfValueBeforeApplyValidator
        implements Validator<ModelChanges<CampaignWithMeaningfulGoalsWithRequiredFields>, Defect> {

    private final Set<Goal> availableCampaignGoals;
    private final Set<Integer> clientCounterIds;
    private final Set<String> enabledFeatures;
    private final CampaignWithMeaningfulGoalsWithRequiredFields currentCampaign;

    private CampaignWithMeaningfulGoalsIsMetrikaSourceOfValueBeforeApplyValidator(@Nullable Set<Goal> availableCampaignGoals,
                                                                                  Set<Integer> clientCounterIds,
                                                                                  Set<String> enabledFeatures,
                                                                                  CampaignWithMeaningfulGoalsWithRequiredFields currentCampaign) {

        this.availableCampaignGoals = availableCampaignGoals;
        this.clientCounterIds = clientCounterIds;
        this.enabledFeatures = enabledFeatures;
        this.currentCampaign = currentCampaign;
    }

    public static CampaignWithMeaningfulGoalsIsMetrikaSourceOfValueBeforeApplyValidator build(
            @Nullable Set<Goal> availableCampaignGoals,
            Set<Integer> clientCounterIds,
            Set<String> enabledFeatures,
            CampaignWithMeaningfulGoalsWithRequiredFields currentCampaign) {
        return new CampaignWithMeaningfulGoalsIsMetrikaSourceOfValueBeforeApplyValidator(
                availableCampaignGoals,
                clientCounterIds, enabledFeatures,
                currentCampaign
        );
    }

    @Override
    public ValidationResult<ModelChanges<CampaignWithMeaningfulGoalsWithRequiredFields>, Defect> apply(
            ModelChanges<CampaignWithMeaningfulGoalsWithRequiredFields> campaignChanges) {
        var vb = ItemValidationBuilder.of(campaignChanges, Defect.class);

        if (CampaignWithMeaningfulGoalsConversionValueBeforeApplyValidator.getCurrencyCode(campaignChanges, currentCampaign) == null) {
            return vb.getResult();
        }

        DbStrategy dbStrategy = nullableNvl(
                campaignChanges.getPropIfChanged(CampaignWithMeaningfulGoalsWithRequiredFields.STRATEGY),
                ifNotNull(currentCampaign, CampaignWithMeaningfulGoalsWithRequiredFields::getStrategy));

        Map<Long, Integer> counterIdByGoalId = StreamEx.of(nvl(availableCampaignGoals, Set.of()))
                .mapToEntry(Goal::getId, Goal::getCounterId)
                .nonNullValues()
                .toMap();

        vb.list(campaignChanges.getPropIfChanged(CampaignWithMeaningfulGoalsWithRequiredFields.MEANINGFUL_GOALS),
                        CampaignWithMeaningfulGoalsWithRequiredFields.MEANINGFUL_GOALS.name())
                .check(maxListSize(MEANINGFUL_GOALS_MAX_COUNT), When.notNull())
                .checkEach(unique(MeaningfulGoal::getGoalId), When.notNull())
                .checkEachBy(goal ->
                        MeaningfulGoalWithIsMetrikaSourceOfValueValidator.build(
                                currentCampaign.getSource(),
                                ifNotNull(dbStrategy, DbStrategyBase::getStrategyName),
                                counterIdByGoalId,
                                clientCounterIds,
                                enabledFeatures).apply(goal),
                        When.notNull());

        return vb.getResult();
    }

    public static class MeaningfulGoalWithIsMetrikaSourceOfValueValidator implements Validator<MeaningfulGoal, Defect> {
        private final CampaignSource campaignSource;
        private final StrategyName strategyName;
        private final Map<Long, Integer> counterIdByGoalId;
        private final Set<Integer> clientCounterIds;
        private final Set<String> enabledFeatures;

        public MeaningfulGoalWithIsMetrikaSourceOfValueValidator(CampaignSource campaignSource,
                                                                 @Nullable StrategyName strategyName,
                                                                 Map<Long, Integer> counterIdByGoalId,
                                                                 Set<Integer> clientCounterIds,
                                                                 Set<String> enabledFeatures) {
            this.campaignSource = campaignSource;
            this.strategyName = strategyName;
            this.counterIdByGoalId = counterIdByGoalId;
            this.clientCounterIds = clientCounterIds;
            this.enabledFeatures = enabledFeatures;
        }

        public static MeaningfulGoalWithIsMetrikaSourceOfValueValidator build(CampaignSource campaignSource,
                                                                              StrategyName strategyName,
                                                                              Map<Long, Integer> counterIdByGoalId,
                                                                              Set<Integer> clientCounterIds,
                                                                              Set<String> enabledFeatures) {
            return new MeaningfulGoalWithIsMetrikaSourceOfValueValidator(campaignSource, strategyName,
                    counterIdByGoalId, clientCounterIds, enabledFeatures);
        }

        @Override
        public ValidationResult<MeaningfulGoal, Defect> apply(MeaningfulGoal meaningfulGoal) {
            var vb = ModelItemValidationBuilder.of(meaningfulGoal);
            validateIsMetrikaSourceOfValue(vb, campaignSource, strategyName, counterIdByGoalId, clientCounterIds,
                    enabledFeatures);
            return vb.getResult();
        }

        public static void validateIsMetrikaSourceOfValue(ModelItemValidationBuilder<MeaningfulGoal> vb,
                                                          CampaignSource source, @Nullable StrategyName strategyName,
                                                          Map<Long, Integer> counterIdByGoalId,
                                                          Set<Integer> clientCounterIds,
                                                          Set<String> enabledFeatures) {
            boolean isCrrStrategy = StrategyName.AUTOBUDGET_CRR == strategyName;
            boolean allowMeaningfulGoalValueFromMetrika =
                    enabledFeatures.contains(FeatureName.ALLOW_MEANINGFUL_GOAL_VALUE_FROM_METRIKA.getName());
            boolean isUnavailableGoal = false;
            if (isUnavailableGoalsAllowed(source, enabledFeatures)) {
                Integer counterId = counterIdByGoalId.get(vb.getResult().getValue().getGoalId());
                isUnavailableGoal = counterId != null && !clientCounterIds.contains(counterId);
            }
            vb.item(MeaningfulGoal.IS_METRIKA_SOURCE_OF_VALUE)
                    .check(notTrue(),
                            When.isTrue(!isCrrStrategy || !allowMeaningfulGoalValueFromMetrika || isUnavailableGoal));
        }
    }
}
