package ru.yandex.direct.core.entity.campaign.service.validation.type.bean.strategy;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.bids.validation.PriceValidator;
import ru.yandex.direct.core.entity.campaign.model.CampOptionsStrategy;
import ru.yandex.direct.core.entity.campaign.model.CampaignSource;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithCustomStrategy;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMeaningfulGoals;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMetrikaCounters;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMobileContent;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithSource;
import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.MeaningfulGoal;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects;
import ru.yandex.direct.core.entity.campaign.service.validation.type.container.CampaignValidationContainer;
import ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.CpmYndxFrontpageAdGroupPriceRestrictions;
import ru.yandex.direct.core.entity.mobileapp.model.MobileApp;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppTrackerTrackingSystem;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContent;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.sitelink.model.SitelinkSet;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.constraint.CommonConstraints;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.time.temporal.ChronoUnit.DAYS;
import static ru.yandex.direct.core.entity.campaign.service.CampaignStrategyUtils.calculateMaximumAvailableBudgetForCpmRestartingStrategyWithCustomPeriod;
import static ru.yandex.direct.core.entity.campaign.service.CampaignStrategyUtils.calculateMinimalAvailableBudgetForCpmRestartingStrategyWithCustomPeriod;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.BY_ALL_GOALS_GOAL_ID;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.CRR_MAX;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.CRR_MIN;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.ENGAGED_SESSION_GOAL_ID;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.PROFITABILITY_MAX;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.PROFITABILITY_MIN;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.RESERVE_RETURN_MAX;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.RESERVE_RETURN_MIN;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.RESERVE_RETURN_STEP;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.ROI_COEF_MIN;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils.CAMPAIGN_TYPES_AVAILABLE_FOR_PAY_FOR_CONVERSION_EXTENDED_MODE;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils.getAutobudgetPayForConversionAvgCpaWarning;
import static ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.allGoalsOptimizationProhibited;
import static ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.avgBidAndBidTogetherAreProhibited;
import static ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.bidLessThanAvgBid;
import static ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.flatCpcStrategyNotSupported;
import static ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.inconsistentStrategyToAdGroupTypes;
import static ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.inconsistentStrategyToCampaignType;
import static ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.inconsistentStrategyToMobileCampaignWithoutTrackingSystem;
import static ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.incorrectReserveReturn;
import static ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.incorrectSetOfMobileGoals;
import static ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.payForConversionDoesNotAllowAllGoals;
import static ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.payForConversionNotSupported;
import static ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.strategyEndDateIsAfterCampaignEndDate;
import static ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.strategyPeriodDaysCountLessThanMin;
import static ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.strategyPeriodDaysCountMoreThanMax;
import static ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.strategyStartDateIsBeforeCampaignStartDate;
import static ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.unableToUseCurrentMeaningfulGoalsForOptimization;
import static ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.weekBudgetLessThan;
import static ru.yandex.direct.core.validation.constraints.Constraints.cpmNotGreaterThan;
import static ru.yandex.direct.core.validation.constraints.Constraints.cpmNotLessThan;
import static ru.yandex.direct.feature.FeatureName.ALLOW_ALL_MEANINGFUL_GOALS_FOR_PAY_FOR_CONVERSION_STRATEGIES;
import static ru.yandex.direct.feature.FeatureName.AUTOBUDGET_STRATEGY_FOR_SMART_ALLOWED;
import static ru.yandex.direct.feature.FeatureName.CPA_PAY_FOR_CONVERSIONS_MOBILE_APPS_ALLOWED;
import static ru.yandex.direct.feature.FeatureName.CPA_STRATEGY_IN_CPM_BANNER_CAMPAIGN_ENABLED;
import static ru.yandex.direct.feature.FeatureName.CRR_STRATEGY_ALLOWED;
import static ru.yandex.direct.feature.FeatureName.FIX_CRR_STRATEGY_ALLOWED;
import static ru.yandex.direct.feature.FeatureName.FLAT_CPC_ADDING_DISABLED;
import static ru.yandex.direct.feature.FeatureName.FLAT_CPC_DISABLED;
import static ru.yandex.direct.feature.FeatureName.RMP_CPI_UNDER_KNOWN_TRACKING_SYSTEM_ONLY;
import static ru.yandex.direct.feature.FeatureName.SOCIAL_ADVERTISING;
import static ru.yandex.direct.feature.FeatureName.UNIVERSAL_CAMPAIGNS_BETA_DISABLED;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.validation.Predicates.not;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicateOfNullable;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.DateConstraints.endDateIsNotBeforeThan;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notGreaterThan;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notLessThan;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notLessThanOrEqualTo;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;

/**
 * Поддержаны проверки не для всех типов кампаний.
 * При добавлении нового типа, нужно подумать о типах стратегийи, которые поддерживает новый тип.
 * И есть ли они в этом валидаторе.
 */
@ParametersAreNonnullByDefault
public class CampaignWithCustomStrategyValidator implements Validator<CampaignWithCustomStrategy, Defect> {
    public static final int MIN_STRATEGY_PERIOD_DAYS_COUNT = 1;
    public static final int MAX_STRATEGY_PERIOD_DAYS_COUNT = 90;
    public static final Set<StrategyName> STRATEGY_TYPES_TO_CHECK_MEANINGFUL_GOALS_IN_CAMPAIGN =
            Set.of(StrategyName.AUTOBUDGET, StrategyName.AUTOBUDGET_ROI, StrategyName.AUTOBUDGET_CRR);

    private static final Logger logger = LoggerFactory.getLogger(CampaignWithCustomStrategyValidator.class);
    private final LocalDate now;
    private final Currency currency;
    //todo: убрать
    private final CampaignWithCustomStrategy campaign;
    private final Set<StrategyName> allowedStrategies;
    private final Set<CampOptionsStrategy> allowedContextStrategies;
    private final Set<CampaignsPlatform> allowedPlatforms;
    private final StrategyValidatorConstants constants;
    private final Supplier<List<BannerWithSystemFields>> campaignBannersSupplier;
    private final Function<List<BannerWithSystemFields>, List<SitelinkSet>> bannersSitelinkSetsFunction;
    private final Set<String> availableFeatures;
    private final Supplier<List<AdGroupSimple>> campaignAdGroupsSupplier;
    private final CampaignValidationContainer campaignValidationContainer;
    private final CpmYndxFrontpageAdGroupPriceRestrictions cpmYndxFrontpageAdGroupPriceRestrictions;
    private final Set<Goal> goals;
    private final boolean requestFromInternalNetwork;
    private final Function<Set<Long>, List<MobileContent>> mobileContentProvider;
    private final Function<Long, MobileApp> mobileAppProvider;
    private final boolean warningOnFlatCpcStrategyUpdate;
    private final DbStrategy oldStrategy;

    public CampaignWithCustomStrategyValidator(Currency currency,
                                               Set<Goal> goals,
                                               Supplier<List<BannerWithSystemFields>> campaignBannersSupplier,
                                               Supplier<List<AdGroupSimple>> campaignAdGroupsSupplier,
                                               Function<List<BannerWithSystemFields>, List<SitelinkSet>> bannersSitelinkSetsFunction,
                                               CampaignWithCustomStrategy campaign,
                                               Set<StrategyName> allowedStrategies,
                                               Set<CampOptionsStrategy> allowedContextStrategies,
                                               Set<CampaignsPlatform> allowedPlatforms,
                                               StrategyValidatorConstants constants,
                                               Set<String> availableFeatures,
                                               CampaignValidationContainer campaignValidationContainer,
                                               @Nullable CpmYndxFrontpageAdGroupPriceRestrictions
                                                       cpmYndxFrontpageAdGroupPriceRestrictions) {
        this(currency, goals, campaignBannersSupplier, campaignAdGroupsSupplier,
                bannersSitelinkSetsFunction, campaign, allowedStrategies, allowedContextStrategies, allowedPlatforms,
                constants, availableFeatures, campaignValidationContainer, cpmYndxFrontpageAdGroupPriceRestrictions,
                false, mobileAppIds -> List.of(), appId -> null, false, null);
    }

    public CampaignWithCustomStrategyValidator(Currency currency,
                                               Set<Goal> goals,
                                               Supplier<List<BannerWithSystemFields>> campaignBannersSupplier,
                                               Supplier<List<AdGroupSimple>> campaignAdGroupsSupplier,
                                               Function<List<BannerWithSystemFields>, List<SitelinkSet>> bannersSitelinkSetsFunction,
                                               CampaignWithCustomStrategy campaign,
                                               Set<StrategyName> allowedStrategies,
                                               Set<CampOptionsStrategy> allowedContextStrategies,
                                               Set<CampaignsPlatform> allowedPlatforms,
                                               StrategyValidatorConstants constants,
                                               Set<String> availableFeatures,
                                               CampaignValidationContainer campaignValidationContainer,
                                               @Nullable CpmYndxFrontpageAdGroupPriceRestrictions
                                                       cpmYndxFrontpageAdGroupPriceRestrictions,
                                               boolean requestFromInternalNetwork,
                                               Function<Set<Long>, List<MobileContent>> mobileContentProvider,
                                               Function<Long, MobileApp> mobileAppProvider,
                                               boolean warningOnFlatCpcStrategyUpdate,
                                               @Nullable DbStrategy oldStrategy) {
        this.currency = currency;
        this.goals = goals;
        this.campaignBannersSupplier = campaignBannersSupplier;
        this.campaignAdGroupsSupplier = campaignAdGroupsSupplier;
        this.bannersSitelinkSetsFunction = bannersSitelinkSetsFunction;
        this.campaign = campaign;
        this.allowedStrategies = allowedStrategies;
        this.allowedContextStrategies = allowedContextStrategies;
        this.allowedPlatforms = allowedPlatforms;
        this.constants = constants;
        this.availableFeatures = availableFeatures;
        this.campaignValidationContainer = campaignValidationContainer;
        this.mobileContentProvider = mobileContentProvider;
        this.now = LocalDate.now();
        this.cpmYndxFrontpageAdGroupPriceRestrictions = cpmYndxFrontpageAdGroupPriceRestrictions;
        this.requestFromInternalNetwork = requestFromInternalNetwork;
        this.mobileAppProvider = mobileAppProvider;
        this.warningOnFlatCpcStrategyUpdate = warningOnFlatCpcStrategyUpdate;
        this.oldStrategy = oldStrategy;
    }

    public static ValidationResult<BigDecimal, Defect> validateSumCommon(BigDecimal sum,
                                                                         BigDecimal minAutobudget,
                                                                         BigDecimal maxAutobudget) {
        return ModelItemValidationBuilder.<BigDecimal, Defect>of(sum)
                .check(notNull())
                .check(notLessThan(minAutobudget))
                .check(notGreaterThan(maxAutobudget))
                .getResult();
    }

    @Override
    public ValidationResult<CampaignWithCustomStrategy, Defect> apply(CampaignWithCustomStrategy campaign) {
        var vb = ModelItemValidationBuilder.of(campaign);
        vb.item(CampaignWithCustomStrategy.STRATEGY)
                .check(notNull())
                .checkBy(this::validateStrategy, When.isValid());
        return vb.getResult();
    }

    public static long getFullDaysCount(LocalDate finish, LocalDate start, LocalDate now) {
        boolean firstDayAlreadyStarted = now.equals(start);
        long daysBetweenStartAndFinish = DAYS.between(start, finish);
        return firstDayAlreadyStarted ? daysBetweenStartAndFinish : daysBetweenStartAndFinish + 1;
    }

    private ValidationResult<DbStrategy, Defect> validateStrategy(DbStrategy dbStrategy) {
        var isCrrStrategyAllowed = availableFeatures.contains(CRR_STRATEGY_ALLOWED.getName());
        // для других типов кампаний, где стратегия не разрешена, валидация сработает и так по allowedStrategies,
        // но для смарт-кампаний autobudget уже внесена в allowedStrategies, так что здесь отдельная проверка
        var isAutobudgetStrategyAllowed = campaign.getType() != CampaignType.PERFORMANCE ||
                availableFeatures.contains(AUTOBUDGET_STRATEGY_FOR_SMART_ALLOWED.getName());
        var flatCpcNotSupported =
                (campaign.getType() != CampaignType.TEXT && campaign.getType() != CampaignType.MOBILE_CONTENT
                        && campaign.getType() != CampaignType.DYNAMIC) || dbStrategy.getStrategyName() != StrategyName.DEFAULT_
                        || dbStrategy.getPlatform() != CampaignsPlatform.BOTH || isUniversalCampaign();
        var isFlatCpcAddingDisabled = availableFeatures.contains(FLAT_CPC_ADDING_DISABLED.getName());
        var isFlatCpcDisabled = availableFeatures.contains(FLAT_CPC_DISABLED.getName());
        ModelItemValidationBuilder<DbStrategy> vb = ModelItemValidationBuilder.of(dbStrategy);
        ValidationResult<StrategyName, Defect> nameResult =
                vb.item(DbStrategy.STRATEGY_NAME)
                        .check(notNull())
                        .check(strategyIsAllowed())
                        .check(strategyIsAllowedForSocialClient())
                        .check(cpvStrategyIsAllowed())
                        .check(notCrrStrategy(), When.isFalse(isCrrStrategyAllowed))
                        .check(notAutobudgetStrategy(), When.isFalse(isAutobudgetStrategyAllowed))
                        .check(strategyIsAllowedForMobileCampaignWithoutTrackingSystem(),
                                When.isTrue(isMobileTrackingSystemRestricted() && isMobileCampaignAndAppWithoutTrackingSystem()))
                        .getResult();
        vb.item(DbStrategy.STRATEGY)
                .check(contextStrategyIsAllowed())
                .check(notFlatCpcStrategy(), When.isTrue(!flatCpcNotSupported && isFlatCpcDisabled))
                .check(notFlatCpcStrategy(), When.isTrue(!flatCpcNotSupported &&
                        isFlatCpcAddingDisabled && !isFlatCpcDisabled && !warningOnFlatCpcStrategyUpdate))
                .weakCheck(notFlatCpcStrategy(), When.isTrue(!flatCpcNotSupported &&
                        isFlatCpcAddingDisabled && !isFlatCpcDisabled && warningOnFlatCpcStrategyUpdate));
        vb.item(DbStrategy.PLATFORM)
                .check(platformIsAllowed());
        vb.item(DbStrategy.STRATEGY_DATA)
                .check(notNull())
                .checkBy(data -> validateStrategyData(data, dbStrategy.getStrategyName()),
                        When.isValidAnd(When.isTrue(!nameResult.hasAnyErrors())));
        return vb.getResult();
    }

    private Constraint<CampaignsPlatform, Defect> platformIsAllowed() {
        return fromPredicate(allowedPlatforms::contains, inconsistentStrategyToCampaignType());
    }

    private Constraint<StrategyName, Defect> strategyIsAllowed() {
        return fromPredicate(allowedStrategies::contains, inconsistentStrategyToCampaignType());
    }

    private Constraint<StrategyName, Defect> strategyIsAllowedForSocialClient() {
        return fromPredicate(strategyName -> !isSocialClient() || !strategyName.equals(StrategyName.AUTOBUDGET_AVG_CPA),
                inconsistentStrategyToCampaignType());
    }

    private Constraint<StrategyName, Defect> strategyIsAllowedForMobileCampaignWithoutTrackingSystem() {
        return fromPredicate(strategyName -> !strategyName.equals(StrategyName.AUTOBUDGET_AVG_CPA) &&
                        !strategyName.equals(StrategyName.AUTOBUDGET_AVG_CPI),
                inconsistentStrategyToMobileCampaignWithoutTrackingSystem());
    }

    private boolean isMobileTrackingSystemRestricted() {
        return availableFeatures.contains(RMP_CPI_UNDER_KNOWN_TRACKING_SYSTEM_ONLY.getName());
    }

    private Constraint<Long, Defect> goalIdIsAllowedForSocialClient() {
        return fromPredicate(goalId -> !isSocialClient() || goalId == null, inconsistentStrategyToCampaignType());
    }

    private boolean isSocialClient() {
        return availableFeatures.contains(SOCIAL_ADVERTISING.getName());
    }

    private Constraint<StrategyName, Defect> cpvStrategyIsAllowed() {
        return fromPredicate(name ->
                !List.of(StrategyName.AUTOBUDGET_AVG_CPV, StrategyName.AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD)
                        .contains(name) || campaignAdGroupsSupplier.get().stream()
                        .allMatch(ag -> ag.getType() == AdGroupType.CPM_VIDEO), inconsistentStrategyToAdGroupTypes());
    }

    private Constraint<StrategyName, Defect> notCrrStrategy() {
        return fromPredicate(not(StrategyName.AUTOBUDGET_CRR::equals), inconsistentStrategyToCampaignType());
    }

    private Constraint<StrategyName, Defect> notAutobudgetStrategy() {
        return fromPredicate(not(StrategyName.AUTOBUDGET::equals), inconsistentStrategyToCampaignType());
    }

    private Constraint<CampOptionsStrategy, Defect> contextStrategyIsAllowed() {
        return fromPredicate(allowedContextStrategies::contains, inconsistentStrategyToCampaignType());
    }

    private Constraint<CampOptionsStrategy, Defect> notFlatCpcStrategy() {
        return fromPredicateOfNullable(CampOptionsStrategy.DIFFERENT_PLACES::equals, flatCpcStrategyNotSupported());
    }

    private ValidationResult<StrategyData, Defect> validateStrategyData(StrategyData strategyData,
                                                                        StrategyName strategyName) {
        ModelItemValidationBuilder<StrategyData> vb = ModelItemValidationBuilder.of(strategyData);

        boolean payForConversionIsSet;

        switch (strategyName) {
            case DEFAULT_:
                vb.item(StrategyData.SUM)
                        .checkBy(this::validateSum, When.notNull());
                break;
            case MIN_PRICE:
            case NO_PREMIUM:
                throw new UnsupportedOperationException(strategyName.name() + " strategy is not supported more");
            case AUTOBUDGET:
                vb.item(StrategyData.SUM)
                        .checkBy(this::validateSum);
                vb.item(StrategyData.BID)
                        .checkBy(bid -> validateBid(bid, strategyData.getSum(), StrategyName.AUTOBUDGET),
                                When.notNull());
                vb.item(StrategyData.GOAL_ID)
                        .check(goalIdIsAllowedForSocialClient())
                        .check(notNull(), When.isTrue(campaign.getType() == CampaignType.PERFORMANCE))
                        .checkBy(goalId -> validateGoalId(goalId, false, strategyName),
                                When.notNull());
                break;
            case AUTOBUDGET_MEDIA:
                throw new UnsupportedOperationException("Validation for AUTOBUDGET_MEDIA strategy is not implemented");
            case AUTOBUDGET_WEEK_BUNDLE:
                vb.item(StrategyData.LIMIT_CLICKS)
                        .checkBy(this::validateLimitClicks);
                vb.item(StrategyData.BID)
                        .checkBy(bid -> validateBid(bid, null, strategyName), When.notNull());
                vb.item(StrategyData.AVG_BID)
                        .checkBy(avgBid -> validateAvgBid(avgBid, null), When.notNull())
                        .checkBy(avgBid -> validateAvgBidAndBidNotTogether(avgBid, strategyData.getBid()),
                                When.notNull());
                break;
            case AUTOBUDGET_AVG_CLICK:
                vb.item(StrategyData.SUM)
                        .checkBy(this::validateSum, When.notNull());
                vb.item(StrategyData.AVG_BID)
                        .checkBy(avgBid -> validateAvgBid(avgBid, strategyData.getSum()))
                        .checkBy(avgBid -> validateAvgBidAndBidNotTogether(avgBid, strategyData.getBid()));
                break;
            case AUTOBUDGET_AVG_CPA:
            case AUTOBUDGET_AVG_CPA_PER_CAMP:
                payForConversionIsSet = isPayForConversionSet(strategyData);

                vb.item(StrategyData.AVG_CPA)
                        .checkBy(avgCpa -> validateAvgCpaOrCpi(avgCpa, strategyData.getSum(), payForConversionIsSet));
                vb.item(StrategyData.GOAL_ID)
                        .checkBy(goalId -> validateGoalId(goalId, payForConversionIsSet, strategyName));
                vb.item(StrategyData.SUM)
                        .checkBy(this::validateSumForAvgCpaOrCpi, When.notNull());
                vb.item(StrategyData.BID)
                        .check(isNull(), When.isTrue(payForConversionIsSet));
                vb.item(StrategyData.BID)
                        .checkBy(bid -> validateBid(bid, strategyData.getSum(), strategyName), When.notNull());
                vb.item(StrategyData.PAY_FOR_CONVERSION)
                        .checkBy(this::validateStrategyConsistentWhenPayForConversionExtendedModeEnabled,
                                When.isTrue(payForConversionIsSet))
                        .checkBy(this::validatePayForConversionExtendedModeEnabledWhenPayForConversionIsTrue,
                                When.isTrue(campaignValidationContainer.getOptions()
                                        .isValidatePayForConversionExtendedModeEnabled()))
                ;
                break;
            case AUTOBUDGET_AVG_CPA_PER_FILTER:
                payForConversionIsSet = isPayForConversionSet(strategyData);

                vb.item(StrategyData.FILTER_AVG_CPA)
                        .checkBy(filterAvgCpa -> validateAvgCpaOrCpi(filterAvgCpa, strategyData.getSum(),
                                campaignValidationContainer.getOptions()
                                        .isValidatePayForConversionInAutobudgetAvgCpaPerFilter()
                                        && payForConversionIsSet));
                vb.item(StrategyData.GOAL_ID)
                        .checkBy(goalId -> validateGoalId(goalId,
                                campaignValidationContainer.getOptions()
                                        .isValidatePayForConversionInAutobudgetAvgCpaPerFilter()
                                        && payForConversionIsSet, strategyName));
                vb.item(StrategyData.SUM)
                        .checkBy(this::validateSumForAvgCpaOrCpi, When.notNull());
                vb.item(StrategyData.BID)
                        .checkBy(bid -> validateBid(bid, strategyData.getSum(), strategyName), When.notNull());
                vb.item(StrategyData.PAY_FOR_CONVERSION)
                        .checkBy(this::validatePayForConversionExtendedModeEnabledWhenPayForConversionIsTrue,
                                When.isTrue(campaignValidationContainer.getOptions()
                                        .isValidatePayForConversionExtendedModeEnabled()));
                break;
            case AUTOBUDGET_AVG_CPC_PER_CAMP:
                vb.item(StrategyData.BID)
                        .checkBy(bid -> validateBid(bid, strategyData.getSum(), strategyName),
                                When.notNull());
                vb.item(StrategyData.SUM)
                        .checkBy(this::validateSum, When.notNull());
                vb.item(StrategyData.AVG_BID)
                        .checkBy(avgBid -> validateAvgBid(avgBid, strategyData.getSum()))
                        .checkBy(avgBid -> validateAvgBidWithBid(avgBid, strategyData.getBid()));
                break;
            case AUTOBUDGET_AVG_CPC_PER_FILTER:
                vb.item(StrategyData.BID)
                        .checkBy(bid -> validateBid(bid, strategyData.getSum(), strategyName),
                                When.notNull());
                vb.item(StrategyData.SUM)
                        .checkBy(this::validateSum, When.notNull());
                vb.item(StrategyData.FILTER_AVG_BID)
                        .checkBy(filterAvgBid -> validateFilterAvgBid(filterAvgBid, strategyData.getSum(),
                                strategyData.getBid()));
                break;
            case AUTOBUDGET_AVG_CPI:
                payForConversionIsSet = isPayForConversionSet(strategyData);

                vb.item(StrategyData.AVG_CPI)
                        .checkBy(avgCpi -> validateAvgCpaOrCpi(avgCpi, strategyData.getSum(), payForConversionIsSet));
                // Для РМП кампании цель = null когда выбираем 'Установки приложения'
                vb.item(StrategyData.GOAL_ID)
                        .checkBy(this::validateMobileContentGoalId, When.notNull());
                vb.item(StrategyData.SUM)
                        .checkBy(this::validateSumForAvgCpaOrCpi, When.notNull());
                vb.item(StrategyData.BID)
                        .check(isNull(), When.isTrue(payForConversionIsSet));
                vb.item(StrategyData.BID)
                        .checkBy(bid -> validateBid(bid, strategyData.getSum(), strategyName),
                                When.notNull());
                vb.item(StrategyData.PAY_FOR_CONVERSION)
                        .checkBy(this::validateStrategyConsistentWhenPayForConversionExtendedModeEnabled,
                                When.isTrue(payForConversionIsSet))
                        .checkBy(this::validatePayForConversionExtendedModeEnabledWhenPayForConversionIsTrue,
                                When.isTrue(campaignValidationContainer.getOptions()
                                        .isValidatePayForConversionExtendedModeEnabled()));
                break;
            case AUTOBUDGET_ROI:
                vb.item(StrategyData.SUM)
                        .checkBy(this::validateSum, When.notNull());
                vb.item(StrategyData.BID)
                        .checkBy(bid -> validateBid(bid, strategyData.getSum(), strategyName), When.notNull());
                vb.item(StrategyData.ROI_COEF)
                        .checkBy(this::validateRoiCoef);
                vb.item(StrategyData.RESERVE_RETURN)
                        .checkBy(this::validateReserveReturn);
                vb.item(StrategyData.PROFITABILITY)
                        .checkBy(this::validateProfitability, When.notNull());
                vb.item(StrategyData.GOAL_ID)
                        .checkBy(goalId -> validateGoalId(goalId, false, strategyName));
                break;
            case AUTOBUDGET_CRR:
                var isFixCrrStrategyAllowed = availableFeatures.contains(FIX_CRR_STRATEGY_ALLOWED.getName());
                vb.item(StrategyData.SUM)
                        .checkBy(this::validateSum, When.notNull());
                vb.item(StrategyData.CRR)
                        .checkBy(this::validateCrr);
                vb.item(StrategyData.GOAL_ID)
                        .checkBy(goalId -> validateGoalId(goalId, isPayForConversionSet(strategyData), strategyName));
                vb.item(StrategyData.PAY_FOR_CONVERSION)
                        .check(CommonConstraints.isNull(), payForConversionNotSupported(),
                                When.isFalse(isFixCrrStrategyAllowed))
                        .checkBy(this::validatePayForConversionExtendedModeEnabledWhenPayForConversionIsTrue,
                                When.isTrue(campaignValidationContainer.getOptions()
                                        .isValidatePayForConversionExtendedModeEnabled()));
                break;
            case CPM_DEFAULT:
                //нет полей
                break;
            case AUTOBUDGET_MAX_REACH:
            case AUTOBUDGET_MAX_IMPRESSIONS:
                vb.item(StrategyData.AVG_CPM)
                        .checkBy(avgCpm -> validateAvgCpm(avgCpm, campaign.getType()));
                vb.item(StrategyData.SUM)
                        .checkBy(this::validateCpmSum);
                break;
            case AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD:
                vb.item(StrategyData.AVG_CPM)
                        .checkBy(avgCpm -> validateAvgCpm(avgCpm, campaign.getType()));
                vb.item(StrategyData.START)
                        .checkBy(this::validateStart);
                vb.item(StrategyData.BUDGET)
                        .checkBy(budget -> validateBudget(budget,
                                calculateMinimalAvailableBudgetForCpmRestartingStrategyWithCustomPeriod(strategyData.getStart(),
                                        strategyData.getFinish(), currency.getCode()),
                                calculateMaximumAvailableBudgetForCpmRestartingStrategyWithCustomPeriod(strategyData.getStart(),
                                        strategyData.getFinish(), currency.getCode())));
                vb.item(StrategyData.FINISH)
                        .checkBy(finish -> validateFinish(finish, strategyData.getStart()));
                break;
            case AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD:
                vb.item(StrategyData.AVG_CPM)
                        .checkBy(avgCpm -> validateAvgCpm(avgCpm, campaign.getType()));
                vb.item(StrategyData.BUDGET)
                        .checkBy(budget -> validateBudget(budget,
                                calculateMinimalAvailableBudgetForCpmRestartingStrategyWithCustomPeriod(strategyData.getStart(),
                                        strategyData.getFinish(), currency.getCode()),
                                calculateMaximumAvailableBudgetForCpmRestartingStrategyWithCustomPeriod(strategyData.getStart(),
                                        strategyData.getFinish(), currency.getCode())));
                vb.item(StrategyData.START)
                        .checkBy(this::validateStart);
                vb.item(StrategyData.FINISH)
                        .checkBy(finish -> validateFinish(finish, strategyData.getStart()));
                break;
            case AUTOBUDGET_AVG_CPV:
                vb.item(StrategyData.AVG_CPV)
                        .checkBy(this::validateAvgCpv);
                vb.item(StrategyData.SUM)
                        .checkBy(this::validateCpmSum);
                break;
            case AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD:
                vb.item(StrategyData.AVG_CPV)
                        .checkBy(this::validateAvgCpv);
                vb.item(StrategyData.BUDGET)
                        .checkBy(budget -> validateBudget(budget,
                                calculateMinimalAvailableBudgetForCpmRestartingStrategyWithCustomPeriod(strategyData.getStart(),
                                        strategyData.getFinish(), currency.getCode()),
                                calculateMaximumAvailableBudgetForCpmRestartingStrategyWithCustomPeriod(strategyData.getStart(),
                                        strategyData.getFinish(), currency.getCode())));
                vb.item(StrategyData.START)
                        .checkBy(this::validateStart);
                vb.item(StrategyData.FINISH)
                        .checkBy(finish -> validateFinish(finish, strategyData.getStart()));
                break;
            default:
                throw new UnsupportedOperationException("Unknown type of strategy "
                        + strategyName.name());
        }
        return vb.getResult();
    }

    private boolean isPayForConversionSet(StrategyData strategyData) {
        return nvl(strategyData.getPayForConversion(), false);
    }

    /**
     * Проверка суммы недельного бюджета
     */
    private ValidationResult<BigDecimal, Defect> validateSum(BigDecimal sum) {
        return validateSumCommon(sum, currency.getMinAutobudget(), currency.getMaxAutobudget());
    }

    /**
     * Проверка суммы недельного бюджета для cpm
     */
    private ValidationResult<BigDecimal, Defect> validateCpmSum(BigDecimal sum) {
        return validateSumCommon(sum, currency.getMinDailyBudgetForPeriod().multiply(BigDecimal.valueOf(7)),
                currency.getMaxAutobudget());
    }

    /**
     * Проверка суммы недельного бюджета для стратегии autobudgetAvgCpa/autobudgetAvgCpi
     */
    private ValidationResult<BigDecimal, Defect> validateSumForAvgCpaOrCpi(BigDecimal sum) {
        var vb = ModelItemValidationBuilder.<BigDecimal, Defect>of(sum);
        vb.check(notNull());
        vb.checkBy(this::validateSum);
        return vb.getResult();
    }

    /**
     * Проверка максимальной ставки
     */
    private ValidationResult<BigDecimal, Defect> validateBid(BigDecimal bid, @Nullable BigDecimal sum,
                                                             StrategyName strategyName) {
        return ModelItemValidationBuilder.<BigDecimal, Defect>of(bid)
                .check(notNull())
                .check(notLessThan(constants.getMinPrice()))
                .check(notGreaterThan(strategyName == StrategyName.AUTOBUDGET_WEEK_BUNDLE
                        ? currency.getMaxPrice()
                        : currency.getMaxAutobudgetBid()))
                .check(notGreaterThan(nvl(sum, BigDecimal.ZERO)), weekBudgetLessThan(),
                        When.isTrue(sum != null))
                .getResult();
    }

    /**
     * Проверка средней цены клика
     */
    private ValidationResult<BigDecimal, Defect> validateAvgBidWithBid(BigDecimal averageBid,
                                                                       @Nullable BigDecimal bid) {
        return ModelItemValidationBuilder.<BigDecimal, Defect>of(averageBid)
                .check(notGreaterThan(nvl(bid, BigDecimal.ZERO)), bidLessThanAvgBid(),
                        When.isTrue(bid != null))
                .getResult();
    }

    private ValidationResult<BigDecimal, Defect> validateAvgBidAndBidNotTogether(BigDecimal averageBid,
                                                                                 @Nullable BigDecimal bid) {
        return ModelItemValidationBuilder.<BigDecimal, Defect>of(averageBid)
                .check(avgBid -> bid == null ? null : avgBidAndBidTogetherAreProhibited())
                .getResult();
    }

    private ValidationResult<BigDecimal, Defect> validateAvgBid(BigDecimal averageBid,
                                                                @Nullable BigDecimal sum) {
        return ModelItemValidationBuilder.<BigDecimal, Defect>of(averageBid)
                .check(notNull())
                .check(notLessThan(constants.getMinAvgPrice()))
                .check(notGreaterThan(currency.getMaxAutobudgetBid()))
                .check(notGreaterThan(nvl(sum, BigDecimal.ZERO)), weekBudgetLessThan(),
                        When.isTrue(sum != null))
                .getResult();
    }

    private ValidationResult<BigDecimal, Defect> validateFilterAvgBid(BigDecimal filterAverageBid,
                                                                      @Nullable BigDecimal sum,
                                                                      @Nullable BigDecimal bid) {
        return ModelItemValidationBuilder.<BigDecimal, Defect>of(filterAverageBid)
                .check(notNull())
                .check(notLessThan(constants.getMinAvgPrice()))
                .check(notGreaterThan(currency.getMaxAutobudgetBid()))
                .check(notGreaterThan(nvl(sum, BigDecimal.ZERO)), weekBudgetLessThan(),
                        When.isTrue(sum != null))
                .check(notGreaterThan(nvl(bid, BigDecimal.ZERO)), bidLessThanAvgBid(), When.isTrue(bid != null))
                .getResult();
    }

    /**
     * Проверка средней цены конверсии (CPA или CPI)
     */
    private ValidationResult<BigDecimal, Defect> validateAvgCpaOrCpi(BigDecimal averageCpa,
                                                                     @Nullable BigDecimal sum,
                                                                     boolean payForConversion) {

        BigDecimal autobudgetPayForConversionAvgCpaWarning = getAutobudgetPayForConversionAvgCpaWarning(
                availableFeatures, currency);
        return ModelItemValidationBuilder.<BigDecimal, Defect>of(averageCpa)
                .check(notNull())
                .check(notLessThan(constants.getMinAvgCpa()))
                .check(notGreaterThan(autobudgetPayForConversionAvgCpaWarning),
                        When.isTrue(payForConversion))
                .check(notGreaterThan(currency.getAutobudgetAvgCpaWarning()), When.isFalse(payForConversion))
                .check(notGreaterThan(nvl(sum, BigDecimal.ZERO)), weekBudgetLessThan(),
                        When.isTrue(sum != null))
                .getResult();
    }

    private ValidationResult<BigDecimal, Defect> validateAvgCpm(BigDecimal averageCpm, CampaignType campaignType) {
        var vb = ModelItemValidationBuilder.<BigDecimal, Defect>of(averageCpm)
                .check(notNull())
                .check(notLessThan(currency.getMinAutobudgetAvgCpm()),
                        When.isTrue(campaignType == CampaignType.CPM_PRICE
                                || campaignType == CampaignType.CPM_YNDX_FRONTPAGE
                                || campaignValidationContainer.getOptions().isForceValidateAvgCpmMin()))
                .check(notGreaterThan(currency.getMaxCpmPrice()))
                .checkBy(price -> new PriceValidator(currency, AdGroupType.CPM_YNDX_FRONTPAGE,
                                cpmYndxFrontpageAdGroupPriceRestrictions).apply(price),
                        When.isTrue(campaignType == CampaignType.CPM_YNDX_FRONTPAGE
                                && cpmYndxFrontpageAdGroupPriceRestrictions != null));
        return vb.getResult();
    }

    private ValidationResult<BigDecimal, Defect> validateAvgCpv(BigDecimal averageCpv) {
        var vb = ModelItemValidationBuilder.<BigDecimal, Defect>of(averageCpv)
                .check(notNull())
                .check(notLessThan(currency.getMinAvgCpv()))
                .check(notGreaterThan(currency.getMaxAvgCpv()));
        return vb.getResult();
    }

    private ValidationResult<LocalDate, Defect> validateStart(LocalDate start) {
        return ModelItemValidationBuilder.<LocalDate, Defect>of(start)
                .check(notNull())
                .check(fromPredicate(startTime -> !startTime.isBefore(campaign.getStartDate()),
                        strategyStartDateIsBeforeCampaignStartDate()),
                        When.isTrue(campaign.getStartDate() != null))
                .getResult();
    }

    private ValidationResult<BigDecimal, Defect> validateBudget(BigDecimal budget, BigDecimal minimalBudget,
                                                                BigDecimal maximumBudget) {
        return ModelItemValidationBuilder.<BigDecimal, Defect>of(budget)
                .check(notNull())
                .check(cpmNotLessThan(minimalBudget, currency.getCode()))
                .check(cpmNotGreaterThan(maximumBudget, currency.getCode()))
                .getResult();
    }

    private ValidationResult<LocalDate, Defect> validateFinish(LocalDate finish, LocalDate start) {

        return ModelItemValidationBuilder.<LocalDate, Defect>of(finish)
                .check(notNull())
                .check(endDateIsNotBeforeThan(start))
                .check(fromPredicate(finishTime -> getFullDaysCount(finishTime, start, now) >= MIN_STRATEGY_PERIOD_DAYS_COUNT,
                        strategyPeriodDaysCountLessThanMin(MIN_STRATEGY_PERIOD_DAYS_COUNT)))
                .check(fromPredicate(finishTime -> getFullDaysCount(finishTime, start, now) <= MAX_STRATEGY_PERIOD_DAYS_COUNT,
                        strategyPeriodDaysCountMoreThanMax(MAX_STRATEGY_PERIOD_DAYS_COUNT)))
                .check(fromPredicate(finishTime -> !finishTime.isAfter(campaign.getEndDate()),
                        strategyEndDateIsAfterCampaignEndDate()),
                        When.isTrue(campaign.getEndDate() != null))
                .getResult();
    }

    /**
     * Проверка возможности использовать оплату за конверсии
     */
    private ValidationResult<Boolean, Defect> validateStrategyConsistentWhenPayForConversionExtendedModeEnabled(
            Boolean payForConversion) {
        var vb = ModelItemValidationBuilder.<Boolean, Defect>of(payForConversion);

        //для расширенной фичи проверяем доступные типы кампаний
        vb.check(fromPredicate(
                value -> CAMPAIGN_TYPES_AVAILABLE_FOR_PAY_FOR_CONVERSION_EXTENDED_MODE.contains(campaign.getType())
                        || campaign.getType() == CampaignType.CPM_BANNER && availableFeatures
                        .contains(CPA_STRATEGY_IN_CPM_BANNER_CAMPAIGN_ENABLED.getName()),
                StrategyDefects.inconsistentStrategyToCampaignType()));

        return vb.getResult();
    }

    private ValidationResult<Boolean, Defect> validatePayForConversionExtendedModeEnabledWhenPayForConversionIsTrue(
            Boolean payForConversion) {

        var mobileAppsAllowed = availableFeatures.contains(CPA_PAY_FOR_CONVERSIONS_MOBILE_APPS_ALLOWED.getName());
        var vb = ModelItemValidationBuilder.<Boolean, Defect>of(payForConversion);

        // https://a.yandex-team.ru/arc_vcs/direct/perl/protected/Direct/Validation/Strategy.pm?rev=r8668070#L765
        vb.check(fromPredicate(it -> (
                        campaign.getType() != CampaignType.MOBILE_CONTENT || mobileAppsAllowed),
                StrategyDefects.payForConversionNotSupported()),
                When.isTrue(nvl(payForConversion, false)));

        return vb.getResult();
    }

    /**
     * Проверка цели.
     * Для РМП проверяется существование цели по захардкоженному списку
     * из {@link ru.yandex.direct.core.entity.metrika.service.MobileGoalsPermissionService}
     * Для остальных идет проверка цели метрики
     *
     * @param goalId                - id цели метрики или специальное значение (все цели, ключевые цели ...)
     * @param payForConversionIsSet - флаг, установлена ли в стратегии "Оплата за конверсии".
     *                              Если стратегия не поддерживает этот параметр, передаем false
     */
    private ValidationResult<Long, Defect> validateGoalId(@Nullable Long goalId,
                                                          boolean payForConversionIsSet,
                                                          StrategyName strategyName) {
        if (campaign.getType() == CampaignType.MOBILE_CONTENT) {
            return validateMobileContentGoalId(goalId);
        }

        if (campaignValidationContainer.getOptions().isCheckDisableAllGoalsOptimizationForDnaFeature()) {
            var hasDisableAllGoalsOptimizationForDnaEnabled = availableFeatures.contains(
                    FeatureName.DISABLE_ALL_GOALS_OPTIMIZATION_FOR_DNA.getName());
            boolean wasAllGoalsGoalId = Optional.ofNullable(oldStrategy)
                    .map(DbStrategy::getStrategyData)
                    .map(StrategyData::getGoalId)
                    .map(BY_ALL_GOALS_GOAL_ID::equals)
                    .orElse(false);
            // если уже стояло BY_ALL_GOALS_GOAL_ID, то разрешаем оставить, чтобы поддержать старые кампании
            boolean allowToKeepAllGoalsOptimization = wasAllGoalsGoalId
                    && campaignValidationContainer.getOptions().isAllowAllGoalsOptimizationForOldCampaigns();
            if (BY_ALL_GOALS_GOAL_ID.equals(goalId)
                    && !allowToKeepAllGoalsOptimization
                    && hasDisableAllGoalsOptimizationForDnaEnabled) {
                return ValidationResult.failed(goalId, allGoalsOptimizationProhibited());
            }
        }

        //выбрано "по всем целям" или id цели среди доступных целей
        if (BY_ALL_GOALS_GOAL_ID.equals(goalId) && !payForConversionIsSet
                || goalId != null && StreamEx.of(goals).anyMatch(goal -> goal.getId().equals(goalId))) {
            return new ValidationResult<>(goalId);
        }

        var allowAllMeaningfulGoalsEnabled =
                availableFeatures.contains(ALLOW_ALL_MEANINGFUL_GOALS_FOR_PAY_FOR_CONVERSION_STRATEGIES.getName());
        //при оплате конверсий выбор "по всем целям" запрещен
        if (payForConversionIsSet && (BY_ALL_GOALS_GOAL_ID.equals(goalId)
                || MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID.equals(goalId)
                && !isMeaningfulGoalsWithPayForConversionAllowed(allowAllMeaningfulGoalsEnabled))) {
            return ValidationResult.failed(goalId, payForConversionDoesNotAllowAllGoals());
        }

        // https://a.yandex-team.ru/arc_vcs/direct/perl/protected/CampaignTools.pm?rev=r9190755#L118
        if (MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID.equals(goalId)
                && (
                !campaignValidationContainer.getOptions().isCheckStrategyForMeaninfulGoalsGoalId() ||
                        (strategyName == StrategyName.AUTOBUDGET_ROI
                                || strategyName == StrategyName.AUTOBUDGET
                                || (strategyName == StrategyName.AUTOBUDGET_CRR
                                && (!payForConversionIsSet || allowAllMeaningfulGoalsEnabled))))) {
            return ModelItemValidationBuilder.<Long, Defect>of(goalId)
                    .check(fromPredicate(
                            goal -> checkMeaningfulGoals(meaningfulGoals ->
                                    StreamEx.of(meaningfulGoals)
                                            .anyMatch(it -> it.getGoalId() != ENGAGED_SESSION_GOAL_ID)
                            ),
                            unableToUseCurrentMeaningfulGoalsForOptimization()),
                            When.isTrue(isMeaningfulGoalsCheckRequired()))
                    //Проверяем, что не более чем 1 mobile_app_id на каждую платформу
                    .check(fromPredicate(
                            goal -> checkMeaningfulGoals(this::checkMobileGoals),
                            incorrectSetOfMobileGoals()),
                            When.isTrue(isMeaningfulGoalsDefined()))
                    .getResult();
        }
        Set<Long> availableGoalIds = new HashSet<>();
        if (campaign instanceof CampaignWithMetrikaCounters) {
            availableGoalIds = getMetrikaGoalIds();
        }

        boolean allGoalsAreAvailable = requestFromInternalNetwork
                && !availableFeatures.contains(UNIVERSAL_CAMPAIGNS_BETA_DISABLED.getName());

        return ModelItemValidationBuilder.<Long, Defect>of(goalId)
                .check(notNull())
                .check(validId())
                .check(fromPredicate(availableGoalIds::contains, objectNotFound()),
                        When.isValidAnd(When.isFalse(campaignValidationContainer.isCopy() || allGoalsAreAvailable)))
                .getResult();
    }

    private boolean isMeaningfulGoalsWithPayForConversionAllowed(boolean allowAllMeaningfulGoalsEnabled) {
        var strategy = campaign.getStrategy().getStrategyName();
        return allowAllMeaningfulGoalsEnabled && strategy == StrategyName.AUTOBUDGET_CRR;
    }

    private boolean isMeaningfulGoalsCheckRequired() {
        var strategyName = campaign.getStrategy().getStrategyName();
        return STRATEGY_TYPES_TO_CHECK_MEANINGFUL_GOALS_IN_CAMPAIGN.contains(strategyName);
    }

    private ValidationResult<Long, Defect> validateMobileContentGoalId(@Nullable Long goalId) {
        // При копировании вообще не проверяем доступность цели, так как она проверяется в
        // CampaignsWithStrategyCheckMetrikaGoalsMediator
        if (campaignValidationContainer.isCopy()) {
            return new ValidationResult<>(goalId);
        }

        var mobileGoalsIds = StreamEx.of(goals)
                .filter(this::isMobileGoal)
                .map(Goal::getId)
                .toSet();

        return ModelItemValidationBuilder.<Long, Defect>of(goalId)
                .check(notNull())
                .check(validId())
                .check(fromPredicate(mobileGoalsIds::contains, objectNotFound()))
                .getResult();
    }

    private Set<Long> getMetrikaGoalIds() {
        return StreamEx.of(goals)
                .remove(this::isMobileGoal)
                .map(Goal::getId)
                .toSet();
    }

    /**
     * Проверка принадлежности цели к РМП с поправкой на null
     */
    private boolean isMobileGoal(Goal goal) {
        return nvl(goal.getIsMobileGoal(), false);
    }

    /**
     * Проверка коэффициента рентабельности
     */
    private ValidationResult<BigDecimal, Defect> validateRoiCoef(BigDecimal roiCoef) {
        return ModelItemValidationBuilder.<BigDecimal, Defect>of(roiCoef)
                .check(notNull())
                .check(notLessThanOrEqualTo(ROI_COEF_MIN))
                .getResult();
    }

    /**
     * Проверка процента возврата в рекламу от сэкономленного бюджета
     */
    private ValidationResult<Long, Defect> validateReserveReturn(Long reserveReturn) {
        return ModelItemValidationBuilder.<Long, Defect>of(reserveReturn)
                .check(notNull())
                .check(fromPredicate(value -> value >= RESERVE_RETURN_MIN && value <= RESERVE_RETURN_MAX
                                && value % RESERVE_RETURN_STEP == 0L,
                        incorrectReserveReturn()))
                .getResult();
    }

    /**
     * Проверка процента доходов, являющихся себестоимостью товаров
     */
    private ValidationResult<BigDecimal, Defect> validateProfitability(BigDecimal profitability) {
        return ModelItemValidationBuilder.<BigDecimal, Defect>of(profitability)
                .check(notNull())
                .check(notLessThan(PROFITABILITY_MIN))
                .check(notGreaterThan(PROFITABILITY_MAX))
                .getResult();
    }

    private ValidationResult<Long, Defect> validateCrr(Long crr) {
        return ModelItemValidationBuilder.<Long, Defect>of(crr)
                .check(notNull())
                .check(notLessThan(CRR_MIN))
                .check(notGreaterThan(CRR_MAX))
                .getResult();
    }

    /**
     * Проверка количества кликов на неделю
     */
    private ValidationResult<Long, Defect> validateLimitClicks(Long limitClicks) {
        return ModelItemValidationBuilder.<Long, Defect>of(limitClicks)
                .check(notNull())
                .check(notLessThan((long) currency.getMinAutobudgetClicksBundle()))
                .check(notGreaterThan(currency.getMaxAutobudgetClicksBundle()))
                .getResult();
    }

    private boolean checkMeaningfulGoals(Predicate<List<MeaningfulGoal>> predicate) {
        return isMeaningfulGoalsDefined() && predicate.test(((CampaignWithMeaningfulGoals) campaign).getMeaningfulGoals());
    }

    private boolean isMeaningfulGoalsDefined() {
        return campaign instanceof CampaignWithMeaningfulGoals
                && ((CampaignWithMeaningfulGoals) campaign).getMeaningfulGoals() != null;
    }

    private boolean checkMobileGoals(List<MeaningfulGoal> meaningfulGoals) {
        var goalIdToMap = StreamEx.of(goals).toMap(Goal::getId, g -> g);

        var mobileAppIds = StreamEx.of(meaningfulGoals)
                .map(mg -> goalIdToMap.get(mg.getGoalId()))
                .filter(g -> Objects.nonNull(g) && Boolean.TRUE.equals(g.getIsMobileGoal()))
                .map(Goal::getMobileAppId)
                .nonNull()
                .collect(Collectors.toSet());
        if (mobileAppIds.isEmpty()) {
            return true;
        } else {
            return StreamEx.of(mobileContentProvider.apply(mobileAppIds))
                    .mapToEntry(MobileContent::getOsType, mc -> mc)
                    .sortedBy(Map.Entry::getKey)
                    .collapseKeys()
                    .noneMatch((osType, contents) -> listToSet(contents).size() > 1);
        }
    }

    private boolean isUniversalCampaign() {
        return campaign instanceof CampaignWithSource && ((CampaignWithSource) campaign).getSource() == CampaignSource.UAC;
    }

    private boolean isMobileCampaignAndAppWithoutTrackingSystem() {
        if (campaign.getType() != CampaignType.MOBILE_CONTENT || !(campaign instanceof CampaignWithMobileContent)) {
            return false;
        }
        if (isUniversalCampaign()) {
            return false;
        }
        MobileApp mobileApp = this.mobileAppProvider.apply(((CampaignWithMobileContent) campaign).getMobileAppId());
        return mobileApp == null || mobileApp.getTrackers().isEmpty() || mobileApp.getTrackers().get(0).getTrackingSystem() == MobileAppTrackerTrackingSystem.OTHER;
    }
}

