package ru.yandex.direct.core.entity.campaign.service.validation.type.bean.strategy;

import java.math.BigDecimal;
import java.time.LocalDate;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.campaign.model.CpmCampaignWithCustomStrategy;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils.isCpmStrategyShouldNotRestart;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils.isCpmStrategyWithCustomPeriodChangedForCheckFinishDate;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils.isCpmStrategyWithCustomPeriodChangedForCheckStartDate;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils.isStrategyAlreadyStartedAndNotFinished;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils.isStrategyNameForCpmStrategyWithCustomPeriod;
import static ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.strategyChangingLimitWasExceeded;
import static ru.yandex.direct.core.validation.constraints.Constraints.cpmNotLessThan;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.DateConstraints.isNotBeforeThan;

@ParametersAreNonnullByDefault
public class CpmCampaignWithCustomStrategyBeforeApplyValidator implements
        Validator<ModelChanges<CpmCampaignWithCustomStrategy>, Defect> {
    public static final int CPM_STRATEGY_MAX_DAILY_CHANGE_COUNT = 4;

    private BigDecimal minimalBudgetWithoutRestarting;
    private CurrencyCode currencyCode;
    private CpmCampaignWithCustomStrategy oldCampaign;
    private LocalDate now;

    private CpmCampaignWithCustomStrategyBeforeApplyValidator(
            BigDecimal minimalBudgetWithoutRestarting,
            CurrencyCode currencyCode,
            CpmCampaignWithCustomStrategy oldCampaign,
            LocalDate now) {
        this.minimalBudgetWithoutRestarting = minimalBudgetWithoutRestarting;
        this.currencyCode = currencyCode;
        this.oldCampaign = oldCampaign;
        this.now = now;
    }

    public static CpmCampaignWithCustomStrategyBeforeApplyValidator build(
            BigDecimal minimalBudgetForCustomPeriod,
            CurrencyCode currencyCode,
            CpmCampaignWithCustomStrategy oldCampaign,
            LocalDate now) {
        return new CpmCampaignWithCustomStrategyBeforeApplyValidator(minimalBudgetForCustomPeriod, currencyCode,
                oldCampaign, now);
    }

    @Override
    public ValidationResult<ModelChanges<CpmCampaignWithCustomStrategy>, Defect> apply(ModelChanges<CpmCampaignWithCustomStrategy> mc) {
        var vb = ItemValidationBuilder.of(mc, Defect.class);

        if (oldCampaign == null) {
            return vb.getResult();
        }

        if (mc.isPropChanged(CpmCampaignWithCustomStrategy.STRATEGY)) {
            vb.item(mc.getChangedProp(CpmCampaignWithCustomStrategy.STRATEGY),
                    CpmCampaignWithCustomStrategy.STRATEGY.name())
                    .checkBy(this::validateStrategy, When.notNull());
        }
        return vb.getResult();
    }

    private ValidationResult<DbStrategy, Defect> validateStrategy(DbStrategy dbStrategy) {
        ModelItemValidationBuilder<DbStrategy> vb = ModelItemValidationBuilder.of(dbStrategy);
        vb.item(DbStrategy.STRATEGY_DATA)
                .checkBy(data -> validateStrategyData(data, dbStrategy.getStrategyName()),
                        When.isValidAnd(When.notNull()));
        return vb.getResult();
    }

    private ValidationResult<StrategyData, Defect> validateStrategyData(StrategyData strategyData,
                                                                        StrategyName strategyName) {
        ModelItemValidationBuilder<StrategyData> vb = ModelItemValidationBuilder.of(strategyData);

        switch (strategyName) {
            case AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD:
            case AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD:
            case AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD:
                validateCpmStrategy(strategyData, vb);
                break;
        }

        return vb.getResult();
    }

    private void validateCpmStrategy(StrategyData strategyData, ModelItemValidationBuilder<StrategyData> vb) {
        vb.check(validateStrategyDataUpdateTimeout(oldCampaign.getStrategy().getStrategyData(), now));

        vb.item(StrategyData.FINISH)
                .checkBy(finish -> validateFinish(finish, now,
                        isCpmStrategyWithCustomPeriodChangedForCheckFinishDate(
                                oldCampaign.getStrategy().getStrategyData(), strategyData)));

        vb.item(StrategyData.START)
                .checkBy(start -> validateStart(start, now,
                        isCpmStrategyWithCustomPeriodChangedForCheckStartDate(
                                oldCampaign.getStrategy().getStrategyData(), strategyData)));
        boolean needToCheckBudget =
                isCpmStrategyShouldNotRestart(oldCampaign.getStrategy().getStrategyData(), strategyData)
                        && isStrategyAlreadyStartedAndNotFinished(oldCampaign.getStrategy().getStrategyData(), now);

        vb.item(StrategyData.BUDGET)
                .checkBy(budget -> validateBudget(budget, minimalBudgetWithoutRestarting, currencyCode,
                        needToCheckBudget));
    }


    private Constraint<StrategyData, Defect> validateStrategyDataUpdateTimeout(StrategyData oldStrategyData,
                                                                               LocalDate now) {
        return Constraint.fromPredicate(data -> !oldStrategyData.getName().equals(data.getName()) ||
                        !(isStrategyNameForCpmStrategyWithCustomPeriod(oldStrategyData) &&
                                isStrategyNameForCpmStrategyWithCustomPeriod(data) &&
                                !now.isBefore(data.getStart()) &&
                                data.getStart().equals(oldStrategyData.getStart()) &&
                                oldStrategyData.getLastUpdateTime() != null &&
                                now.equals(oldStrategyData.getLastUpdateTime().toLocalDate()) &&
                                oldStrategyData.getDailyChangeCount() >= CPM_STRATEGY_MAX_DAILY_CHANGE_COUNT),
                strategyChangingLimitWasExceeded(CPM_STRATEGY_MAX_DAILY_CHANGE_COUNT));
    }

    private ValidationResult<LocalDate, Defect> validateStart(LocalDate start,
                                                              LocalDate now, boolean isStrategyChanged) {
        return ModelItemValidationBuilder.<LocalDate, Defect>of(start)
                .check(notNull())
                .check(isNotBeforeThan(now), When.isTrue(isStrategyChanged))
                .getResult();
    }

    private ValidationResult<LocalDate, Defect> validateFinish(LocalDate finish, LocalDate now,
                                                               boolean isStrategyChangedIgnoringAutoProlongation) {
        return ModelItemValidationBuilder.<LocalDate, Defect>of(finish)
                .check(isNotBeforeThan(now), When.notNullAnd(When.isTrue(isStrategyChangedIgnoringAutoProlongation)))
                .getResult();
    }

    private ValidationResult<BigDecimal, Defect> validateBudget(BigDecimal budget,
                                                                BigDecimal minimalBudgetWithoutRestarting,
                                                                CurrencyCode currencyCode,
                                                                Boolean isNotRestarting) {
        return ModelItemValidationBuilder.<BigDecimal, Defect>of(budget)
                .check(cpmNotLessThan(minimalBudgetWithoutRestarting, currencyCode),
                        When.notNullAnd(When.isTrue(isNotRestarting)))
                .getResult();
    }

}
