package ru.yandex.direct.core.entity.campaign.service.validation.type.disabled;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.service.validation.type.container.CampaignValidationContainer;
import ru.yandex.direct.core.entity.campaign.service.validation.type.update.AbstractCampaignUpdateValidationTypeSupport;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.campaign.model.CommonCampaign.IS_RECOMMENDATIONS_MANAGEMENT_ENABLED;
import static ru.yandex.direct.validation.result.DefectIds.FORBIDDEN_TO_CHANGE;

public abstract class AbstractCampaignUpdateDisabledFieldValidationTypeSupport<C extends BaseCampaign>
    extends AbstractCampaignUpdateValidationTypeSupport<C> {

    private static final Logger logger = LoggerFactory.getLogger(AbstractCampaignUpdateDisabledFieldValidationTypeSupport.class);

    @Autowired(required = false)
    private DisabledFieldsDataProvider disabledDataProvider;

    abstract DisabledField getCheckField();

    abstract public ModelProperty<? super C, ?> getChanged(AppliedChanges<C> changes);

    @Override
    public ValidationResult<List<C>, Defect> validate(
            CampaignValidationContainer container,
            ValidationResult<List<C>, Defect> vr,
            Map<Integer, AppliedChanges<C>> appliedChangesForValidModelChanges)
    {
        return new ListValidationBuilder<>(vr)
                .checkEachBy((index, campaign) -> validate(
                        container,
                        appliedChangesForValidModelChanges.get(index),
                        campaign))
                .getResult();
    }

    public ValidationResult<C, Defect> validate(
            CampaignValidationContainer container,
            AppliedChanges<C> changes,
            C campaign
    ) {
        var vb = ModelItemValidationBuilder.of(campaign);
        var disabledInfoContainer = container.getDisabledFieldsInfoContainer();
        var cid = campaign.getId();
        if (container.getOptions().isAutoApply()) {
            // Автоматическое изменение не должно блокироваться прошлым автоматическим применением
            return vb.getResult();
        }
        if (campaign instanceof CommonCampaign && Boolean.FALSE.equals(changes
                .castModelUp(CommonCampaign.class)
                .getNewValue(IS_RECOMMENDATIONS_MANAGEMENT_ENABLED))) {
            // Если опция выключена, то проверять нечего
            return vb.getResult();
        }
        if (campaign instanceof CommonCampaign && changes.castModelUp(CommonCampaign.class)
                .changed(IS_RECOMMENDATIONS_MANAGEMENT_ENABLED)) {
            // Если включаем опцию, то автоматических применений еще не было
            // Если выключаем фичу, то уже пользователь может сам управлять данными
            return vb.getResult();
        }
        if (!disabledInfoContainer.canObjectHaveDisabledFields(cid)) {
            return vb.getResult();
        }
        var changed = getChanged(changes);
        if (changed == null) {
            return vb.getResult();
        }
        var disabledInfo = disabledInfoContainer.getDisabledFieldInfo(cid, getCheckField(),
                cids -> calculateDisabledFields(container, cids));
        if (disabledInfo != null) {
            vb.item(changed).check(Constraint.fromPredicateOfNullable(f -> false,
                    new Defect(FORBIDDEN_TO_CHANGE)));
        }
        return vb.getResult();
    }

    private Map<Long, Map<DisabledField, Object>> calculateDisabledFields(
            CampaignValidationContainer container,
            Collection<Long> cids
    ) {
        if (disabledDataProvider == null) {
            logger.error("Валидация дизейбленных полей не работает из-за недостатка зависимостей в сервисе.");
            // Если видим такой лог, то нужно добавить зависимость на direct.lib.communication в сервис
            return Collections.emptyMap();
        }
        return disabledDataProvider.getCampaignDisabledData(
                container.getClientId(),
                container.getOperatorUid(),
                cids);
    }

}
