package ru.yandex.direct.core.entity.campaign.service.validation.type.update;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.model.BrandSurveyStatus;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithBrandLift;
import ru.yandex.direct.core.entity.campaign.service.CampaignBudgetReachService;
import ru.yandex.direct.core.entity.campaign.service.CampaignWithBrandLiftExperimentsService;
import ru.yandex.direct.core.entity.campaign.service.validation.type.bean.CampaignWithBrandLiftValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.type.container.CampaignValidationContainer;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingConditionBase;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.metrika.client.MetrikaClientException;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstraints.metrikaReturnsResultWithErrors;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.brandLiftCantBeChanged;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.brandLiftExperimentSegmentsCantBeChanged;
import static ru.yandex.direct.core.validation.defects.RightsDefects.noRights;
import static ru.yandex.direct.feature.FeatureName.BRAND_LIFT;
import static ru.yandex.direct.feature.FeatureName.BRAND_LIFT_HIDDEN;
import static ru.yandex.direct.metrika.client.model.response.RetargetingCondition.Type.AB_SEGMENT;
import static ru.yandex.direct.multitype.service.validation.type.ValidationTypeSupportUtils.forEachModelChangesAddDefectIfFieldChanged;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithBrandLiftUpdateValidationTypeSupport extends AbstractCampaignUpdateValidationTypeSupport<CampaignWithBrandLift> {

    private static final Logger logger =
            LoggerFactory.getLogger(CampaignWithBrandLiftUpdateValidationTypeSupport.class);

    private final FeatureService featureService;
    private final RetargetingConditionService retargetingConditionService;
    private final CampaignBudgetReachService campaignBudgetReachService;
    private final CampaignWithBrandLiftExperimentsService campaignWithBrandLiftExperimentsService;

    @Autowired
    public CampaignWithBrandLiftUpdateValidationTypeSupport(FeatureService featureService,
                                                            RetargetingConditionService retargetingConditionService,
                                                            CampaignBudgetReachService campaignBudgetReachService,
                                                            CampaignWithBrandLiftExperimentsService campaignWithBrandLiftExperimentsService) {
        this.featureService = featureService;
        this.retargetingConditionService = retargetingConditionService;
        this.campaignBudgetReachService = campaignBudgetReachService;
        this.campaignWithBrandLiftExperimentsService = campaignWithBrandLiftExperimentsService;
    }

    @Override
    public Class<CampaignWithBrandLift> getTypeClass() {
        return CampaignWithBrandLift.class;
    }

    @Override
    public ValidationResult<List<ModelChanges<CampaignWithBrandLift>>, Defect> preValidate(
            CampaignValidationContainer container, ValidationResult<List<ModelChanges<CampaignWithBrandLift>>,
            Defect> vr) {

        if (!featureService.isEnabledForClientId(container.getClientId(), FeatureName.AB_SEGMENTS)) {
            forEachModelChangesAddDefectIfFieldChanged(vr, CampaignWithBrandLift.AB_SEGMENT_GOAL_IDS);
            forEachModelChangesAddDefectIfFieldChanged(vr, CampaignWithBrandLift.SECTION_IDS);
        }

        // изменения флага is_brand_lift_hidden делаем только из кода операции обновления,
        // основываясь на остальных полях, поэтому запрещаем любые изменения извне
        forEachModelChangesAddDefectIfFieldChanged(vr, CampaignWithBrandLift.IS_BRAND_LIFT_HIDDEN);

        return vr;
    }

    @Override
    public ValidationResult<List<ModelChanges<CampaignWithBrandLift>>, Defect> validateBeforeApply(
            CampaignValidationContainer container,
            ValidationResult<List<ModelChanges<CampaignWithBrandLift>>, Defect> vr,
            Map<Long, CampaignWithBrandLift> unmodifiedModels) {

        boolean brandLiftEnabledForClientId = featureService.isEnabledForClientId(container.getClientId(), BRAND_LIFT);
        boolean brandLiftHiddenEnabledForOperator =
                featureService.isEnabled(container.getOperatorUid(), BRAND_LIFT_HIDDEN);

        Validator<ModelChanges<CampaignWithBrandLift>, Defect> validator = changes -> {
            ItemValidationBuilder<ModelChanges<CampaignWithBrandLift>, Defect> vb =
                    ItemValidationBuilder.of(changes);

            CampaignWithBrandLift model = unmodifiedModels.get(changes.getId());

            if (changes.isPropChanged(CampaignWithBrandLift.BRAND_SURVEY_ID)) {
                vb.item(changes.getPropIfChanged(CampaignWithBrandLift.BRAND_SURVEY_ID),
                        CampaignWithBrandLift.BRAND_SURVEY_ID.name())
                        .check(brandLiftCantBeChangedToAnother(model, brandLiftHiddenEnabledForOperator),
                                When.notNull())
                        .check(brandLiftCanBeChangedByClient(model, brandLiftEnabledForClientId));
            }

            vb.item(changes.getPropIfChanged(CampaignWithBrandLift.AB_SEGMENT_GOAL_IDS),
                    CampaignWithBrandLift.AB_SEGMENT_GOAL_IDS.name())
                    .check(Constraint.fromPredicate(Objects::isNull, brandLiftExperimentSegmentsCantBeChanged()),
                            When.isTrue(brandLiftHiddenEnabledForOperator && model.getIsBrandLiftHidden()));

            vb.item(changes.getPropIfChanged(CampaignWithBrandLift.SECTION_IDS),
                    CampaignWithBrandLift.SECTION_IDS.name())
                    .check(Constraint.fromPredicate(sectionIds -> sectionIds == null || sectionIds.isEmpty(),
                            brandLiftExperimentSegmentsCantBeChanged()),
                            When.isTrue(brandLiftHiddenEnabledForOperator && model.getIsBrandLiftHidden()));
            return vb.getResult();
        };

        var vb = new ListValidationBuilder<>(vr);
        vb.checkEachBy(validator, When.isValid());
        return vb.getResult();
    }

    @Override
    public ValidationResult<List<CampaignWithBrandLift>, Defect> validate(
            CampaignValidationContainer container,
            ValidationResult<List<CampaignWithBrandLift>, Defect> vr,
            Map<Integer, AppliedChanges<CampaignWithBrandLift>> appliedChangesForValidModelChanges) {

        ClientId clientId = container.getClientId();
        ListValidationBuilder<CampaignWithBrandLift, Defect> vb = new ListValidationBuilder<>(vr);

        List<ru.yandex.direct.metrika.client.model.response.RetargetingCondition> clientAbSegments;
        try {
            Map<Long, List<ru.yandex.direct.metrika.client.model.response.RetargetingCondition>> goalsByUids =
                    container.getMetrikaClient().getAbSegmentGoals();
            clientAbSegments = getClientAbSegments(goalsByUids);
        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            logger.warn("Got an exception when querying for metrika goals for clientId: " + clientId, e);
            vb.checkEach(metrikaReturnsResultWithErrors(), When.notNull());
            return vb.getResult();
        }

        Map<Long, List<Long>> retargetingConditionIdToGoalIds = retargetingConditionIdToGoalIds(container,
                vr.getValue());

        Map<Long, BrandSurveyStatus> brandStatusByCampaignId =
                campaignBudgetReachService.getBrandStatusForCampaigns(container.getShard(), clientId,
                        mapList(vr.getValue(),
                                CampaignWithBrandLift::getId));

        Map<Long, Boolean> isBrandLiftHiddenByCampaignId = appliedChangesForValidModelChanges.values()
                .stream()
                .collect(Collectors.toMap(change -> change.getModel().getId(),
                        change -> nvl(change.getOldValue(CampaignWithBrandLift.IS_BRAND_LIFT_HIDDEN), false)));

        Map<Long, String> brandSurveyIdByCampaignId = StreamEx.of(appliedChangesForValidModelChanges.values())
                .mapToEntry(
                        change -> change.getModel().getId(),
                        change -> change.getOldValue(CampaignWithBrandLift.BRAND_SURVEY_ID))
                .filterValues(Objects::nonNull)
                .toMap();

        return vb
                .checkEachBy(new CampaignWithBrandLiftValidator(
                        retargetingConditionIdToGoalIds,
                        brandStatusByCampaignId,
                        clientAbSegments,
                        isBrandLiftHiddenByCampaignId,
                        brandSurveyIdByCampaignId))
                .getResult();
    }

    /*
     * проходит по всем кампания, вытаскивает по ab_segment_ret_cond_id условия ретаргетинга, получает из них целей
     * и возвращает мапу из ab_segment_ret_cond_id в список id целей
     */
    private Map<Long, List<Long>> retargetingConditionIdToGoalIds(CampaignValidationContainer container,
                                                                  List<CampaignWithBrandLift> campaigns) {
        List<Long> retargetingConditionIds = StreamEx.of(campaigns)
                .map(CampaignWithBrandLift::getAbSegmentRetargetingConditionId)
                .nonNull()
                .toList();

        List<RetargetingConditionBase> retargetingCondition =
                retargetingConditionService.get(container.getClientId(), container.getOperatorUid(),
                        retargetingConditionIds);

        Map<Long, RetargetingConditionBase> retagetingConditionById = listToMap(retargetingCondition,
                RetargetingConditionBase::getId);

        Map<Long, List<Long>> retargetingConditionIdToGoalIds = EntryStream.of(retagetingConditionById)
                .mapValues(CampaignWithBrandLiftUpdateValidationTypeSupport::abSegmentGoalIdsFromRetargetingCondition)
                .toMap();

        return retargetingConditionIdToGoalIds;
    }

    private static List<ru.yandex.direct.metrika.client.model.response.RetargetingCondition> getClientAbSegments(Map<Long,
            List<ru.yandex.direct.metrika.client.model.response.RetargetingCondition>> goalsByUids) {
        return EntryStream.of(goalsByUids)
                .values()
                .flatMap(Collection::stream)
                .filter(x -> x.getType() == AB_SEGMENT)
                .toList();
    }

    private static List<Long> abSegmentGoalIdsFromRetargetingCondition(RetargetingConditionBase retargetingCondition) {
        return StreamEx.of(retargetingCondition.getRules())
                .map(Rule::getGoals)
                .flatMap(StreamEx::of)
                .filter(g -> g.getType() == GoalType.AB_SEGMENT)
                .map(Goal::getId)
                .toList();
    }

    private static Constraint<String, Defect> brandLiftCanBeChangedByClient(CampaignWithBrandLift model,
                                                                            boolean brandLiftEnabledForClientId) {

        return checkIfBrandLiftActuallyChanged(model, c -> brandLiftEnabledForClientId, noRights());
    }


    private static Constraint<String, Defect> brandLiftCantBeChangedToAnother(CampaignWithBrandLift model,
                                                                              boolean brandLiftHiddenEnabledForOperator) {

        return checkIfBrandLiftActuallyChanged(model, c -> c.getShows() == null || c.getShows().equals(0L) ||
                        !brandLiftHiddenEnabledForOperator && c.getIsBrandLiftHidden(),
                brandLiftCantBeChanged());
    }


    private static Constraint<String, Defect> checkIfBrandLiftActuallyChanged(
            CampaignWithBrandLift model,
            Predicate<CampaignWithBrandLift> predicate, Defect<Void> defect) {

        return Constraint.fromPredicateOfNullable(newBrandSurveyId ->
                !brandLiftIsActuallyChanged(model, newBrandSurveyId) || predicate.test(model), defect);
    }

    private static boolean brandLiftIsActuallyChanged(CampaignWithBrandLift model, String newBrandSurveyId) {
        String oldBrandSurveyId = CampaignWithBrandLift.BRAND_SURVEY_ID.get(model);
        return !Objects.equals(oldBrandSurveyId, newBrandSurveyId);
    }
}
