package ru.yandex.direct.core.entity.campaign.service.validation.type.update;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.model.BroadMatch;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithBroadMatch;
import ru.yandex.direct.core.entity.campaign.service.validation.type.bean.CampaignWithBroadMatchBeforeApplyValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.type.container.CampaignValidationContainer;
import ru.yandex.direct.core.entity.metrika.service.MetrikaGoalsService;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalBase;
import ru.yandex.direct.core.validation.defects.Defects;
import ru.yandex.direct.metrika.client.MetrikaClientException;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithBroadMatchUpdateValidationTypeSupport extends AbstractCampaignUpdateValidationTypeSupport<CampaignWithBroadMatch> {
    private static final Logger logger =
            LoggerFactory.getLogger(CampaignWithBroadMatchUpdateValidationTypeSupport.class);

    private final MetrikaGoalsService metrikaGoalsService;

    public CampaignWithBroadMatchUpdateValidationTypeSupport(MetrikaGoalsService metrikaGoalsService) {
        this.metrikaGoalsService = metrikaGoalsService;
    }

    @Override
    public Class<CampaignWithBroadMatch> getTypeClass() {
        return CampaignWithBroadMatch.class;
    }

    @Override
    public ValidationResult<List<ModelChanges<CampaignWithBroadMatch>>, Defect> validateBeforeApply(
            CampaignValidationContainer container,
            ValidationResult<List<ModelChanges<CampaignWithBroadMatch>>, Defect> vr,
            Map<Long, CampaignWithBroadMatch> unmodifiedModels) {
        var clientId = container.getClientId();
        var operatorUid = container.getOperatorUid();
        var vb = new ListValidationBuilder<>(vr);

        var validCampaigns = ValidationResult.getValidItems(vr);
        var campaignIdToCampaignType = new HashMap<Long, CampaignType>(validCampaigns.size());
        validCampaigns.forEach(mc -> campaignIdToCampaignType.put(
                mc.getId(), ifNotNull(unmodifiedModels.get(mc.getId()), CampaignWithBroadMatch::getType)));

        Map<Long, Set<Long>> availableGoalIdsForCampaignId;
        try {
            Map<Long, Set<Goal>> availableGoalsForCampaignId = metrikaGoalsService
                    .getAvailableBroadMatchesForCampaignId(operatorUid, clientId, campaignIdToCampaignType);
            availableGoalIdsForCampaignId = EntryStream.of(availableGoalsForCampaignId)
                    .mapValues(x -> listToSet(x, GoalBase::getId))
                    .toMap();
        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            logger.warn("Got an exception when querying for metrika goals for clientId: " + clientId, e);
            vb.checkEach(Constraint.fromPredicate(mc ->
                            mc.getPropIfChanged(CampaignWithBroadMatch.BROAD_MATCH) != null &&
                                    isInCampaignUsedSameGoalId(unmodifiedModels, mc),
                    Defects.metrikaReturnsResultWithErrors()), When.isValid());
            return vb.getResult();
        }

        vb.checkEachBy(mc -> CampaignWithBroadMatchBeforeApplyValidator
                .build(availableGoalIdsForCampaignId.get(mc.getId()), unmodifiedModels.get(mc.getId()))
                .apply(mc), When.isValid());

        return vb.getResult();
    }

    private boolean isInCampaignUsedSameGoalId(
            Map<Long, CampaignWithBroadMatch> unmodifiedModels,
            ModelChanges<CampaignWithBroadMatch> mc) {
        BroadMatch broadMatch = mc.getPropIfChanged(
                CampaignWithBroadMatch.BROAD_MATCH);
        Long newGoalId = ifNotNull(broadMatch, BroadMatch::getBroadMatchGoalId);
        Long oldGoalId = Optional.ofNullable(unmodifiedModels.get(mc.getId()))
                .map(CampaignWithBroadMatch::getBroadMatch)
                .map(BroadMatch::getBroadMatchGoalId)
                .orElse(null);
        return Objects.equals(newGoalId, oldGoalId);
    }
}
