package ru.yandex.direct.core.entity.campaign.service.validation.type.update;

import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithContentLanguage;
import ru.yandex.direct.core.entity.campaign.service.validation.type.bean.CampaignWithContentLanguageValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.type.container.CampaignValidationContainer;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.GeoTreeFactory;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyMap;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.ROLES_ALLOWED_TO_CHANGE_CONTENT_LANGUAGE;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithContentLanguageUpdateValidationTypeSupport extends AbstractCampaignUpdateValidationTypeSupport<CampaignWithContentLanguage> {
    private final RbacService rbacService;
    private final GeoTreeFactory geoTreeFactory;
    private final AdGroupRepository adGroupRepository;

    @Autowired
    public CampaignWithContentLanguageUpdateValidationTypeSupport(RbacService rbacService,
                                                                  GeoTreeFactory geoTreeFactory,
                                                                  AdGroupRepository adGroupRepository) {
        this.rbacService = rbacService;
        this.geoTreeFactory = geoTreeFactory;
        this.adGroupRepository = adGroupRepository;
    }

    @Override
    public Class<CampaignWithContentLanguage> getTypeClass() {
        return CampaignWithContentLanguage.class;
    }

    @Override
    public ValidationResult<List<ModelChanges<CampaignWithContentLanguage>>, Defect> preValidate(
            CampaignValidationContainer container,
            ValidationResult<List<ModelChanges<CampaignWithContentLanguage>>, Defect> vr) {
        RbacRole operatorRole = rbacService.getUidRole(container.getOperatorUid());
        boolean hasRights = ROLES_ALLOWED_TO_CHANGE_CONTENT_LANGUAGE.contains(operatorRole);

        Map<Long, List<AdGroupSimple>> adGroupsSimpleByCampaignId = emptyMap();
        if (hasRights) {
            int shard = container.getShard();

            List<Long> campaignWithChangedLanguageIds = StreamEx.of(vr.getValue())
                    .filter(CampaignWithContentLanguageUpdateValidationTypeSupport::isContentLanguageChanged)
                    .map(ModelChanges::getId)
                    .toList();

            adGroupsSimpleByCampaignId =
                    adGroupRepository.getAdGroupSimpleByCampaignsIds(shard, campaignWithChangedLanguageIds);
        }

        var validator = new CampaignWithContentLanguageValidator(getRussianGeoTree(), adGroupsSimpleByCampaignId, hasRights);
        return new ListValidationBuilder<>(vr)
                .checkEachBy(validator::apply)
                .getResult();
    }

    private static boolean isContentLanguageChanged(ModelChanges<CampaignWithContentLanguage> mc) {
        return mc.isPropChanged(CampaignWithContentLanguage.CONTENT_LANGUAGE);
    }

    private GeoTree getRussianGeoTree() {
        // Для валидации соответствия регионов с языком кампании и баннеров используется российское геодерево
        return geoTreeFactory.getRussianGeoTree();
    }
}
