package ru.yandex.direct.core.entity.campaign.service.validation.type.update;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.net.NetAcl;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.model.BannerWithSitelinks;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithCustomStrategy;
import ru.yandex.direct.core.entity.campaign.service.validation.type.MobileContentServiceHelper;
import ru.yandex.direct.core.entity.campaign.service.validation.type.bean.strategy.CampaignWithCustomStrategyValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.type.bean.strategy.StrategyValidatorConstantsBuilder;
import ru.yandex.direct.core.entity.campaign.service.validation.type.container.CampaignValidationContainer;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.currency.service.CpmYndxFrontpageCurrencyService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.metrika.container.CampaignTypeWithCounterIds;
import ru.yandex.direct.core.entity.metrika.service.campaigngoals.CampaignGoalsService;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.sitelink.model.SitelinkSet;
import ru.yandex.direct.core.entity.sitelink.service.SitelinkSetService;
import ru.yandex.direct.core.util.CoreHttpUtil;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.metrika.client.MetrikaClientException;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static ru.yandex.direct.core.entity.campaign.converter.CampaignConverter.toCampaignTypeWithCounterIds;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstraints.metrikaReturnsResultWithErrors;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils.ALL_CAMPAIGNS_PLATFORMS;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils.ALL_CAMP_OPTIONS_STRATEGIES;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils.CAMPAIGN_TO_CONTEXT_STRATEGY_TYPE;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils.CAMPAIGN_TO_PLATFORM_TYPE;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils.CAMPAIGN_TO_STRATEGY_TYPE;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithCustomStrategyUpdateValidationTypeSupport
        extends AbstractCampaignUpdateValidationTypeSupport<CampaignWithCustomStrategy> {
    private static final Logger logger =
            LoggerFactory.getLogger(CampaignWithCustomStrategyUpdateValidationTypeSupport.class);

    private final ClientService clientService;
    private final FeatureService featureService;
    private final SitelinkSetService sitelinkSetService;
    private final BannerTypedRepository bannerRepository;
    private final CpmYndxFrontpageCurrencyService cpmYndxFrontpageCurrencyService;
    private final AdGroupRepository adGroupRepository;
    private final CampaignGoalsService campaignGoalsService;
    private final NetAcl netAcl;
    private final MobileContentServiceHelper mobileContentServiceHelper;

    @Autowired
    public CampaignWithCustomStrategyUpdateValidationTypeSupport(
            ClientService clientService,
            BannerTypedRepository bannerRepository,
            FeatureService featureService,
            SitelinkSetService sitelinkSetService,
            CpmYndxFrontpageCurrencyService cpmYndxFrontpageCurrencyService,
            AdGroupRepository adGroupRepository,
            CampaignGoalsService campaignGoalsService,
            NetAcl netAcl,
            MobileContentServiceHelper mobileContentServiceHelper) {
        this.clientService = clientService;
        this.bannerRepository = bannerRepository;
        this.featureService = featureService;
        this.sitelinkSetService = sitelinkSetService;
        this.cpmYndxFrontpageCurrencyService = cpmYndxFrontpageCurrencyService;
        this.adGroupRepository = adGroupRepository;
        this.campaignGoalsService = campaignGoalsService;
        this.netAcl = netAcl;
        this.mobileContentServiceHelper = mobileContentServiceHelper;
    }

    @Override
    public Class<CampaignWithCustomStrategy> getTypeClass() {
        return CampaignWithCustomStrategy.class;
    }

    @Override
    public ValidationResult<List<CampaignWithCustomStrategy>, Defect> validate(
            CampaignValidationContainer container,
            ValidationResult<List<CampaignWithCustomStrategy>, Defect> vr,
            Map<Integer, AppliedChanges<CampaignWithCustomStrategy>> appliedChangesForValidModelChanges) {
        var vb = new ListValidationBuilder<>(vr);
        int shard = container.getShard();
        var clientId = container.getClientId();
        var operatorUid = container.getOperatorUid();
        var campaigns = vr.getValue();

        var enabledForClientId = featureService.getEnabledForClientId(clientId);
        Map<Long, CampaignTypeWithCounterIds> campaignTypesWithCountersByCampaignId = listToMap(campaigns,
                CampaignWithCustomStrategy::getId,
                campaign -> toCampaignTypeWithCounterIds(campaign, enabledForClientId,
                        container.getGoalIdToCounterIdForCampaignsWithoutCounterIds()));

        var currency = clientService.getWorkCurrency(clientId);

        var restrictionsByCampaignIds =
                cpmYndxFrontpageCurrencyService.getPriceDataByCampaigns(campaigns, shard, clientId);

        Map<Long, Set<Goal>> availableGoalsByCampaignId;
        try {
            availableGoalsByCampaignId = campaignGoalsService.getAvailableGoalsForCampaignId(
                    operatorUid,
                    clientId,
                    campaignTypesWithCountersByCampaignId,
                    container.getMetrikaClient()
            );
        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            logger.warn("Got an exception when querying for metrika counters for clientId: " + clientId, e);
            vb.checkEach(metrikaReturnsResultWithErrors());
            return vb.getResult();
        }

        boolean requestFromInternalNetwork = Optional.ofNullable(CoreHttpUtil.getRemoteAddressFromAuthOrDefault())
                .map(netAcl::isInternalIp)
                .orElse(false);

        return vb
                .checkEachBy((index, campaign) -> {
                    Long campaignId = campaign.getId();
                    CampaignType campaignType =
                            campaignTypesWithCountersByCampaignId.get(campaignId).getCampaignType();
                    var oldStrategy = appliedChangesForValidModelChanges.get(index)
                            .getOldValue(CampaignWithCustomStrategy.STRATEGY);
                    return new CampaignWithCustomStrategyValidator(
                            currency,
                            availableGoalsByCampaignId.getOrDefault(campaignId, Set.of()),
                            getCampaignBannersSupplier(shard, campaignId),
                            getCampaignAdGroupsSupplier(shard, campaignId),
                            getBannersSiteLinkSetsFunction(clientId),
                            campaign,
                            CAMPAIGN_TO_STRATEGY_TYPE.getOrDefault(campaignType, emptySet()),
                            CAMPAIGN_TO_CONTEXT_STRATEGY_TYPE.getOrDefault(campaign.getType(),
                                    ALL_CAMP_OPTIONS_STRATEGIES),
                            CAMPAIGN_TO_PLATFORM_TYPE.getOrDefault(campaign.getType(), ALL_CAMPAIGNS_PLATFORMS),
                            StrategyValidatorConstantsBuilder.build(campaignType, currency),
                            enabledForClientId,
                            container,
                            restrictionsByCampaignIds.get(campaignId),
                            requestFromInternalNetwork,
                            mobileAppIds -> mobileContentServiceHelper.getMobileContents(clientId, mobileAppIds),
                            mobileAppId -> mobileContentServiceHelper.getMobileApp(clientId, mobileAppId),
                            true,
                            oldStrategy
                    ).apply(campaign);
                })
                .getResult();
    }

    private Supplier<List<BannerWithSystemFields>> getCampaignBannersSupplier(int shard, Long campaignId) {
        return () -> bannerRepository.getBannersByCampaignIdsAndClass(shard, List.of(campaignId),
                BannerWithSystemFields.class);
    }

    private Supplier<List<AdGroupSimple>> getCampaignAdGroupsSupplier(int shard, Long campaignId) {
        return () -> adGroupRepository.getAdGroupSimpleByCampaignsIds(shard, List.of(campaignId))
                .getOrDefault(campaignId, emptyList());
    }

    private Function<List<BannerWithSystemFields>, List<SitelinkSet>> getBannersSiteLinkSetsFunction(ClientId clientId) {
        return banners -> getSitelinkSetsForBanner(clientId, banners);
    }

    private List<SitelinkSet> getSitelinkSetsForBanner(ClientId clientId,
                                                       Collection<BannerWithSystemFields> banners) {

        var bannerSiteLinksSetIds = StreamEx.of(banners)
                .select(BannerWithSitelinks.class)
                .filter(b -> b.getSitelinksSetId() != null)
                .map(BannerWithSitelinks::getSitelinksSetId)
                .toSet();

        if (bannerSiteLinksSetIds.isEmpty()) {
            return Collections.emptyList();
        }

        return sitelinkSetService.getSitelinkSets(bannerSiteLinksSetIds, clientId,
                LimitOffset.maxLimited());
    }
}
