package ru.yandex.direct.core.entity.campaign.service.validation.type.update;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.model.CampaignWithExperiments;
import ru.yandex.direct.core.entity.campaign.service.validation.type.bean.CampaignWithExperimentsUpdatePreValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.type.bean.CampaignWithExperimentsUpdateValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.type.container.CampaignValidationContainer;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingConditionBase;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.metrika.client.MetrikaClientException;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstraints.metrikaReturnsResultWithErrors;
import static ru.yandex.direct.feature.FeatureName.AB_SEGMENTS;
import static ru.yandex.direct.feature.FeatureName.EXPERIMENT_RET_CONDITIONS_CREATING_ON_TEXT_CAMPAIGNS_MODIFY_IN_JAVA_FOR_DNA;
import static ru.yandex.direct.metrika.client.model.response.RetargetingCondition.Type.AB_SEGMENT;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithExperimentsUpdateValidationTypeSupport
        extends AbstractCampaignUpdateValidationTypeSupport<CampaignWithExperiments> {

    private static final Logger logger =
            LoggerFactory.getLogger(CampaignWithExperimentsUpdateValidationTypeSupport.class);

    private final FeatureService featureService;
    private final RetargetingConditionService retargetingConditionService;

    @Autowired
    public CampaignWithExperimentsUpdateValidationTypeSupport(
            FeatureService featureService, RetargetingConditionService retargetingConditionService) {
        this.featureService = featureService;
        this.retargetingConditionService = retargetingConditionService;
    }

    @Override
    public Class<CampaignWithExperiments> getTypeClass() {
        return CampaignWithExperiments.class;
    }

    @Override
    public ValidationResult<List<ModelChanges<CampaignWithExperiments>>, Defect> preValidate(
            CampaignValidationContainer container,
            ValidationResult<List<ModelChanges<CampaignWithExperiments>>, Defect> vr) {

        ClientId clientId = container.getClientId();
        Set<String> enabledFeaturesForClient = featureService.getEnabledForClientId(clientId);
        boolean abSegmentsEnabledForClientId = enabledFeaturesForClient.contains(AB_SEGMENTS.getName());
        boolean experimentRetargetingConditionsAutoCreatingEnabledForClientId = enabledFeaturesForClient
                .contains(EXPERIMENT_RET_CONDITIONS_CREATING_ON_TEXT_CAMPAIGNS_MODIFY_IN_JAVA_FOR_DNA.getName());

        var vb = new ListValidationBuilder<>(vr);
        return vb
                .checkEachBy(new CampaignWithExperimentsUpdatePreValidator(abSegmentsEnabledForClientId,
                        experimentRetargetingConditionsAutoCreatingEnabledForClientId))
                .getResult();

    }

    @Override
    public ValidationResult<List<CampaignWithExperiments>, Defect> validate(
            CampaignValidationContainer container,
            ValidationResult<List<CampaignWithExperiments>, Defect> vr,
            Map<Integer, AppliedChanges<CampaignWithExperiments>> appliedChangesForValidModelChanges) {

        ClientId clientId = container.getClientId();
        Set<String> enabledFeaturesForClient = featureService.getEnabledForClientId(clientId);

        var vb = new ListValidationBuilder<>(vr);

        List<RetargetingCondition> clientConditions =
                retargetingConditionService.getRetargetingConditions(clientId, null,
                        LimitOffset.maxLimited());
        Set<Long> clientConditionIds = listToSet(clientConditions, RetargetingConditionBase::getId);

        boolean experimentRetargetingConditionsAutoCreatingEnabledForClientId = enabledFeaturesForClient
                .contains(EXPERIMENT_RET_CONDITIONS_CREATING_ON_TEXT_CAMPAIGNS_MODIFY_IN_JAVA_FOR_DNA.getName());
        List<ru.yandex.direct.metrika.client.model.response.RetargetingCondition> clientAbSegments;
        if (experimentRetargetingConditionsAutoCreatingEnabledForClientId) {
            try {
                Map<Long, List<ru.yandex.direct.metrika.client.model.response.RetargetingCondition>> goalsByUids =
                        container.getMetrikaClient().getAbSegmentGoals();

                clientAbSegments = getClientAbSegments(goalsByUids);
            } catch (MetrikaClientException | InterruptedRuntimeException e) {
                logger.warn("Got an exception when querying for metrika goals for clientId: " + clientId, e);
                vb.checkEach(metrikaReturnsResultWithErrors(), When.notNull());
                return vb.getResult();
            }
        } else {
            clientAbSegments = emptyList();
        }

        vb.checkEachBy((index, campaign) -> CampaignWithExperimentsUpdateValidator.build(
                clientConditionIds,
                clientAbSegments,
                appliedChangesForValidModelChanges.get(index))
                .apply(campaign));

        return vb.getResult();
    }

    private List<ru.yandex.direct.metrika.client.model.response.RetargetingCondition> getClientAbSegments(Map<Long,
            List<ru.yandex.direct.metrika.client.model.response.RetargetingCondition>> goalsByUids) {
        return EntryStream.of(goalsByUids)
                .values()
                .flatMap(Collection::stream)
                .filter(x -> x.getType() == AB_SEGMENT)
                .toList();
    }
}
