package ru.yandex.direct.core.entity.campaign.service.validation.type.update;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.net.NetAcl;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMeaningfulGoalsWithRequiredFields;
import ru.yandex.direct.core.entity.campaign.model.MeaningfulGoal;
import ru.yandex.direct.core.entity.campaign.service.validation.type.bean.meaningfulgoals.CampaignWithMeaningfulGoalsConversionValueBeforeApplyValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.type.bean.meaningfulgoals.CampaignWithMeaningfulGoalsGoalIdBeforeApplyValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.type.bean.meaningfulgoals.CampaignWithMeaningfulGoalsIsMetrikaSourceOfValueBeforeApplyValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.type.container.CampaignValidationContainer;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.metrika.container.CampaignTypeWithCounterIds;
import ru.yandex.direct.core.entity.metrika.service.campaigngoals.CampaignGoalsService;
import ru.yandex.direct.core.util.CoreHttpUtil;
import ru.yandex.direct.core.validation.defects.Defects;
import ru.yandex.direct.metrika.client.MetrikaClientException;
import ru.yandex.direct.metrika.client.model.response.CounterInfoDirect;
import ru.yandex.direct.metrika.client.model.response.UserCountersExtended;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptySet;
import static ru.yandex.direct.core.entity.campaign.converter.CampaignConverter.toCampaignTypeWithCounterIds;
import static ru.yandex.direct.feature.FeatureName.UNIVERSAL_CAMPAIGNS_BETA_DISABLED;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nullableNvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithMeaningfulGoalsUpdateValidationTypeSupport extends AbstractCampaignUpdateValidationTypeSupport<CampaignWithMeaningfulGoalsWithRequiredFields> {
    private static final Logger logger =
            LoggerFactory.getLogger(CampaignWithMeaningfulGoalsUpdateValidationTypeSupport.class);

    private final CampaignGoalsService campaignGoalsService;
    private final NetAcl netAcl;
    private final FeatureService featureService;

    @Autowired
    public CampaignWithMeaningfulGoalsUpdateValidationTypeSupport(CampaignGoalsService campaignGoalsService,
                                                                  NetAcl netAcl,
                                                                  FeatureService featureService) {
        this.campaignGoalsService = campaignGoalsService;
        this.netAcl = netAcl;
        this.featureService = featureService;
    }

    @Override
    public Class<CampaignWithMeaningfulGoalsWithRequiredFields> getTypeClass() {
        return CampaignWithMeaningfulGoalsWithRequiredFields.class;
    }

    @Override
    public ValidationResult<List<ModelChanges<CampaignWithMeaningfulGoalsWithRequiredFields>>, Defect> validateBeforeApply(
            CampaignValidationContainer container,
            ValidationResult<List<ModelChanges<CampaignWithMeaningfulGoalsWithRequiredFields>>, Defect> vr,
            Map<Long, CampaignWithMeaningfulGoalsWithRequiredFields> unmodifiedModels) {
        var clientId = container.getClientId();
        var vb = new ListValidationBuilder<>(vr);

        var validCampaigns = ValidationResult.getValidItems(vr);
        var modelChangesWithChangedMeaningfulGoals = StreamEx.of(validCampaigns)
                .filter(mc -> mc.isPropChanged(CampaignWithMeaningfulGoalsWithRequiredFields.MEANINGFUL_GOALS))
                .mapToEntry(mc -> mc.getChangedProp(CampaignWithMeaningfulGoalsWithRequiredFields.MEANINGFUL_GOALS))
                .nonNullValues()
                .removeValues(List::isEmpty)
                .keys()
                .toList();

        Set<String> enabledFeatures = featureService.getEnabledForClientId(clientId);

        vb
                .checkEachBy(campaign -> CampaignWithMeaningfulGoalsConversionValueBeforeApplyValidator
                        .build(unmodifiedModels.get(campaign.getId()))
                        .apply(campaign));

        try {

            var campaignTypesWithCountersByCampaignId =
                    new HashMap<Long, CampaignTypeWithCounterIds>(modelChangesWithChangedMeaningfulGoals.size());
            modelChangesWithChangedMeaningfulGoals.forEach(mc -> campaignTypesWithCountersByCampaignId.put(mc.getId(),
                    ifNotNull(unmodifiedModels.get(mc.getId()),
                            campaign -> toCampaignTypeWithCounterIds(
                                    campaign, getCurrentMetrikaCounterIds(unmodifiedModels, mc), enabledFeatures))));
            var availableGoalsForCampaignId =
                    campaignGoalsService.getAvailableGoalsForCampaignId(
                            container.getOperatorUid(),
                            clientId,
                            campaignTypesWithCountersByCampaignId,
                            container.getMetrikaClient()
                    );
            Set<Integer> clientCounterIds = StreamEx.of(container.getMetrikaClient()
                            .getUsersCountersNumExtendedByCampaignCounterIds())
                    .flatCollection(UserCountersExtended::getCounters)
                    .map(CounterInfoDirect::getId)
                    .toSet();

            boolean requestFromInternalNetwork = Optional.ofNullable(CoreHttpUtil.getRemoteAddressFromAuthOrDefault())
                    .map(netAcl::isInternalIp)
                    .orElse(false);

            boolean allGoalsAreAvailable = requestFromInternalNetwork
                    && !enabledFeatures.contains(UNIVERSAL_CAMPAIGNS_BETA_DISABLED.getName());

            vb
                    .checkEachBy(campaign -> CampaignWithMeaningfulGoalsIsMetrikaSourceOfValueBeforeApplyValidator
                            .build(availableGoalsForCampaignId.get(campaign.getId()), clientCounterIds,
                                    enabledFeatures, unmodifiedModels.get(campaign.getId()))
                            .apply(campaign))
                    .checkEachBy(campaign -> CampaignWithMeaningfulGoalsGoalIdBeforeApplyValidator
                            .build(availableGoalsForCampaignId.get(campaign.getId()), allGoalsAreAvailable,
                                    unmodifiedModels.get(campaign.getId()))
                            .apply(campaign)
                    );

        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            logger.warn("Got an exception when querying for metrika goals for clientId: " + clientId, e);
            vb.checkEach(Constraint.fromPredicate(mc -> isInCampaignUsedSameGoalIds(unmodifiedModels, mc),
                    Defects.metrikaReturnsResultWithErrors()), When.notNull());
        }

        return vb.getResult();
    }

    private boolean isInCampaignUsedSameGoalIds(
            Map<Long, CampaignWithMeaningfulGoalsWithRequiredFields> unmodifiedModels,
            ModelChanges<CampaignWithMeaningfulGoalsWithRequiredFields> mc) {
        List<MeaningfulGoal> goals = mc.getPropIfChanged(
                CampaignWithMeaningfulGoalsWithRequiredFields.MEANINGFUL_GOALS);
        Set<Long> newGoalIds = listToSet(goals, MeaningfulGoal::getGoalId);
        Set<Long> oldGoalIds = Optional.ofNullable(unmodifiedModels.get(mc.getId()))
                .map(CampaignWithMeaningfulGoalsWithRequiredFields::getMeaningfulGoals)
                .map(list -> listToSet(list, MeaningfulGoal::getGoalId)).orElse(Set.of());

        return oldGoalIds.equals(newGoalIds);
    }

    private static Set<Long> getCurrentMetrikaCounterIds(
            Map<Long, CampaignWithMeaningfulGoalsWithRequiredFields> unmodifiedModels,
            ModelChanges<CampaignWithMeaningfulGoalsWithRequiredFields> mc) {
        List<Long> newMetrikaCountersIfChanged =
                mc.getPropIfChanged(CampaignWithMeaningfulGoalsWithRequiredFields.METRIKA_COUNTERS);

        List<Long> oldMetrikaCounters = ifNotNull(unmodifiedModels.get(mc.getId()),
                CampaignWithMeaningfulGoalsWithRequiredFields::getMetrikaCounters);

        List<Long> currentMetrikaCounters = nullableNvl(newMetrikaCountersIfChanged, oldMetrikaCounters);

        if (currentMetrikaCounters == null) {
            return emptySet();
        }

        return Set.copyOf(currentMetrikaCounters);
    }
}
