package ru.yandex.direct.core.entity.campaign.service.validation.type.update;

import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.model.CampaignWithAdvancedGeoTargeting;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithOptionalMeaningfulGoalsValuesFromMetrika;
import ru.yandex.direct.core.entity.campaign.service.validation.type.bean.CampaignWithAdvancedGeoTargetingUpdatePreValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.type.container.CampaignValidationContainer;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.feature.FeatureName.ADVANCED_GEOTARGETING;
import static ru.yandex.direct.multitype.service.validation.type.ValidationTypeSupportUtils.forEachModelChangesAddDefectIfFieldChanged;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithOptionalAdvancedGeoTargetingUpdateValidationTypeSupport extends AbstractCampaignUpdateValidationTypeSupport<CampaignWithAdvancedGeoTargeting> {
    private final FeatureService featureService;

    public CampaignWithOptionalAdvancedGeoTargetingUpdateValidationTypeSupport(FeatureService featureService) {
        this.featureService = featureService;
    }

    @Override
    public Class<CampaignWithAdvancedGeoTargeting> getTypeClass() {
        return CampaignWithAdvancedGeoTargeting.class;
    }

    @Override
    public ValidationResult<List<ModelChanges<CampaignWithAdvancedGeoTargeting>>, Defect> preValidate(
            CampaignValidationContainer container,
            ValidationResult<List<ModelChanges<CampaignWithAdvancedGeoTargeting>>, Defect> vr
    ) {
        var advancedGeoTargeting = featureService.isEnabledForClientId(container.getClientId(), ADVANCED_GEOTARGETING);
        if (!advancedGeoTargeting) {
            forEachModelChangesAddDefectIfFieldChanged(vr,
                    CampaignWithOptionalMeaningfulGoalsValuesFromMetrika.USE_CURRENT_REGION);
            forEachModelChangesAddDefectIfFieldChanged(vr,
                    CampaignWithOptionalMeaningfulGoalsValuesFromMetrika.USE_REGULAR_REGION);
        } else {
            var vb = new ListValidationBuilder<>(vr);
            vb.checkEachBy(campaign -> new CampaignWithAdvancedGeoTargetingUpdatePreValidator().apply(campaign));
        }
        return vr;
    }

    @Override
    public ValidationResult<List<CampaignWithAdvancedGeoTargeting>, Defect> validate(
            CampaignValidationContainer container,
            ValidationResult<List<CampaignWithAdvancedGeoTargeting>, Defect> vr) {

        var vb = new ListValidationBuilder<>(vr);

        vb.checkEachBy(campaign -> {
            var vbi = ModelItemValidationBuilder.of(campaign);
            vbi.item(CampaignWithAdvancedGeoTargeting.HAS_EXTENDED_GEO_TARGETING)
                    .check(notNull());
            return vbi.getResult();
        });

        return vb.getResult();
    }

    @Override
    public ValidationResult<List<ModelChanges<CampaignWithAdvancedGeoTargeting>>, Defect> validateBeforeApply(
            CampaignValidationContainer container,
            ValidationResult<List<ModelChanges<CampaignWithAdvancedGeoTargeting>>, Defect> vr,
            Map<Long, CampaignWithAdvancedGeoTargeting> unmodifiedModels) {

        boolean advancedGeoTargeting =
                featureService.isEnabledForClientId(container.getClientId(), ADVANCED_GEOTARGETING);

        if (!advancedGeoTargeting) {
            return new ListValidationBuilder<>(vr).getResult();
        }

        Validator<ModelChanges<CampaignWithAdvancedGeoTargeting>, Defect> validator = changes -> {
            ItemValidationBuilder<ModelChanges<CampaignWithAdvancedGeoTargeting>, Defect> vb =
                    ItemValidationBuilder.of(changes);

            CampaignWithAdvancedGeoTargeting model = unmodifiedModels.get(changes.getId());

            if (changes.isPropChanged(CampaignWithAdvancedGeoTargeting.USE_CURRENT_REGION)
                    && changes.isPropChanged(CampaignWithAdvancedGeoTargeting.USE_REGULAR_REGION)) {
                vb.check(checkCorrectChange());
            } else {
                vb.item(changes.getPropIfChanged(CampaignWithAdvancedGeoTargeting.USE_CURRENT_REGION),
                        CampaignWithAdvancedGeoTargeting.USE_CURRENT_REGION.name())
                        .check(checkCorrectOnlyOneChange(model.getUseRegularRegion()), When.isValid());

                vb.item(changes.getPropIfChanged(CampaignWithAdvancedGeoTargeting.USE_REGULAR_REGION),
                        CampaignWithAdvancedGeoTargeting.USE_REGULAR_REGION.name())
                        .check(checkCorrectOnlyOneChange(model.getUseCurrentRegion()), When.isValid());
            }
            return vb.getResult();

        };

        var vb = new ListValidationBuilder<>(vr);
        vb.checkEachBy(validator, When.isValid());
        return vb.getResult();
    }


    public static Constraint<ModelChanges<CampaignWithAdvancedGeoTargeting>, Defect> checkCorrectChange() {
        return Constraint.fromPredicate(c -> {
            var useCurrentRegion = c.getPropIfChanged(CampaignWithAdvancedGeoTargeting.USE_CURRENT_REGION);
            var useRegularRegion = c.getPropIfChanged(CampaignWithAdvancedGeoTargeting.USE_REGULAR_REGION);
            return !(Boolean.FALSE.equals(useCurrentRegion) && Boolean.FALSE.equals(useRegularRegion));
        }, invalidValue());
    }

    public static Constraint<Boolean, Defect> checkCorrectOnlyOneChange(
            boolean otherValue
    ) {
        return Constraint.fromPredicate(c -> otherValue || c, invalidValue());
    }
}
