package ru.yandex.direct.core.entity.campaign.service.validation.type.update;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.model.CampaignWithOrganization;
import ru.yandex.direct.core.entity.campaign.service.validation.type.container.CampaignValidationContainer;
import ru.yandex.direct.core.entity.organization.model.Organization;
import ru.yandex.direct.core.entity.organizations.service.OrganizationService;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.organizations.validation.OrganizationConstraints.checkStatus;
import static ru.yandex.direct.core.entity.organizations.validation.OrganizationDefects.eitherPermalinkOrChainCanBeFilled;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithOrganizationUpdateValidationTypeSupport
        extends AbstractCampaignUpdateValidationTypeSupport<CampaignWithOrganization> {

    private final OrganizationService organizationService;

    @Autowired
    public CampaignWithOrganizationUpdateValidationTypeSupport(OrganizationService organizationService) {
        this.organizationService = organizationService;
    }

    @Override
    public Class<CampaignWithOrganization> getTypeClass() {
        return CampaignWithOrganization.class;
    }

    @Override
    public ValidationResult<List<ModelChanges<CampaignWithOrganization>>, Defect> preValidate(
            CampaignValidationContainer container,
            ValidationResult<List<ModelChanges<CampaignWithOrganization>>, Defect> vr) {
        return new ListValidationBuilder<>(vr)
                .checkEachBy(preValidator())
                .getResult();
    }

    static Validator<ModelChanges<CampaignWithOrganization>, Defect> preValidator() {
        return changes -> {
            ItemValidationBuilder<ModelChanges<CampaignWithOrganization>, Defect> vb =
                    ItemValidationBuilder.of(changes);
            vb.item(changes.getPropIfChanged(CampaignWithOrganization.DEFAULT_PERMALINK_ID),
                    CampaignWithOrganization.DEFAULT_PERMALINK_ID.name())
                    .check(validId(), When.notNull());
            vb.item(changes.getPropIfChanged(CampaignWithOrganization.DEFAULT_CHAIN_ID),
                    CampaignWithOrganization.DEFAULT_CHAIN_ID.name())
                    .check(validId(), When.notNull());
            vb.check(fromPredicate(
                    CampaignWithOrganizationUpdateValidationTypeSupport::validatePermalinkChainUniqueness,
                    eitherPermalinkOrChainCanBeFilled()),
                    When.isValid());
            return vb.getResult();
        };
    }

    /**
     * Проверяет, что из двух полей {@link CampaignWithOrganization#getDefaultPermalinkId()} и
     * {@link CampaignWithOrganization#getDefaultChainId()} заполнено максимум одно.
     */
    private static boolean validatePermalinkChainUniqueness(ModelChanges<CampaignWithOrganization> changes) {
        return changes.getPropIfChanged(CampaignWithOrganization.DEFAULT_PERMALINK_ID) == null
                || changes.getPropIfChanged(CampaignWithOrganization.DEFAULT_CHAIN_ID) == null;
    }

    @Override
    public ValidationResult<List<CampaignWithOrganization>, Defect> validate(
            CampaignValidationContainer container,
            ValidationResult<List<CampaignWithOrganization>, Defect> vr,
            Map<Integer, AppliedChanges<CampaignWithOrganization>> appliedChangesForValidModelChanges) {

        Set<Long> changedPermalinkIds = filterAndMapToSet(appliedChangesForValidModelChanges.values(),
                ac -> ac.changedAndNotDeleted(CampaignWithOrganization.DEFAULT_PERMALINK_ID),
                ac -> ac.getNewValue(CampaignWithOrganization.DEFAULT_PERMALINK_ID));

        if (changedPermalinkIds.isEmpty()) {
            return vr;
        }

        Map<Long, Organization> organizations =
                organizationService.getClientOrganizations(changedPermalinkIds, container.getClientId());

        return new ListValidationBuilder<>(vr)
                .checkEachBy(validator(organizations, changedPermalinkIds, container.isCopy()))
                .getResult();
    }

    static Validator<CampaignWithOrganization, Defect> validator(
            Map<Long, Organization> clientOrganizations,
            Set<Long> changedPermalinkIds,
            boolean isCopy
    ) {
        return campaign -> {
            ModelItemValidationBuilder<CampaignWithOrganization> vb =
                    ModelItemValidationBuilder.of(campaign);
            vb.item(CampaignWithOrganization.DEFAULT_PERMALINK_ID)
                    .check(checkStatus(clientOrganizations, isCopy),
                            When.valueIs(changedPermalinkIds::contains));
            // todo: проверять, что сеть существует в Справочнике и подходит нам
            //  (подходит = она непустая, или у неё есть хотя бы одна неопубликованная организация, или как-то ещё)
            return vb.getResult();
        };
    }

}
