package ru.yandex.direct.core.entity.campaign.service.validation.type.update;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithCampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPackageStrategy;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.campaign.service.validation.type.bean.CampaignWithPackageStrategyUpdatePreValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.type.bean.CampaignWithPackageStrategyUpdateValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.type.container.CampaignValidationContainer;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.strategy.model.BaseStrategy;
import ru.yandex.direct.core.entity.strategy.model.CommonStrategy;
import ru.yandex.direct.core.entity.strategy.model.StrategyWithCampaignIds;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.utils.CollectionUtils;
import ru.yandex.direct.utils.CommonUtils;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.common.db.PpcPropertyNames.MAX_NUMBER_OF_CIDS_ABLE_TO_LINK_TO_PACKAGE_STRATEGY;
import static ru.yandex.direct.core.entity.strategy.service.StrategyConstants.DEFAULT_MAX_NUMBERS_OF_CIDS_ABLE_TO_LINK_TO_PACKAGE_STRATEGY;
import static ru.yandex.direct.feature.FeatureName.PACKAGE_STRATEGIES_STAGE_TWO;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithPackageStrategyUpdateValidationTypeSupport extends AbstractCampaignUpdateValidationTypeSupport<CampaignWithPackageStrategy> {

    private final FeatureService featureService;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final CampaignTypedRepository campaignTypedRepository;

    @Autowired
    public CampaignWithPackageStrategyUpdateValidationTypeSupport(FeatureService featureService,
                                                                  PpcPropertiesSupport ppcPropertiesSupport,
                                                                  CampaignTypedRepository campaignTypedRepository) {
        this.featureService = featureService;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.campaignTypedRepository = campaignTypedRepository;
    }

    @Override
    public ValidationResult<List<ModelChanges<CampaignWithPackageStrategy>>, Defect> preValidate(
            CampaignValidationContainer container,
            ValidationResult<List<ModelChanges<CampaignWithPackageStrategy>>, Defect> vr) {
        var isPackageStrategiesStageTwoEnabled = featureService.isEnabledForClientId(container.getClientId(),
                PACKAGE_STRATEGIES_STAGE_TWO);

        if (!isPackageStrategiesStageTwoEnabled) {
            return vr;
        }

        var strategyFromModelChangesById = StreamEx.of(vr.getValue())
                .map(mc -> mc.getPropIfChanged(CampaignWithPackageStrategy.STRATEGY_ID))
                .filter(CommonUtils::isValidId)
                .distinct()
                .mapToEntry(container::getPackageStrategy)
                .nonNullValues()
                .toMap();

        var changingCampaignsById = getChangingCampaignsById(container.getShard(), vr.getValue());

        var typesOfChangingCampaignsById = EntryStream.of(changingCampaignsById).toMap();

        var idsOfCampaignsWithPublicStrategy = EntryStream.of(changingCampaignsById)
                .mapValues(c -> container.getPackageStrategy(c.getStrategyId()))
                .filterValues(CampaignWithPackageStrategyUpdateValidationTypeSupport::isPublic)
                .keys()
                .toSet();

        var campaignIdToPackageStrategy = EntryStream.of(changingCampaignsById)
                .mapValues(c -> container.getPackageStrategy(c.getStrategyId()))
                .toMap();

        var preValidator = createPreValidator(
                strategyFromModelChangesById,
                typesOfChangingCampaignsById,
                idsOfCampaignsWithPublicStrategy,
                campaignIdToPackageStrategy);

        return new ListValidationBuilder<>(vr)
                .checkEachBy(preValidator)
                .getResult();
    }

    @Override
    public ValidationResult<List<CampaignWithPackageStrategy>, Defect> validate(
            CampaignValidationContainer container,
            ValidationResult<List<CampaignWithPackageStrategy>, Defect> vr) {
        boolean isPackageStrategiesStageTwoEnabled = featureService.isEnabledForClientId(container.getClientId(),
                PACKAGE_STRATEGIES_STAGE_TWO);

        if (!isPackageStrategiesStageTwoEnabled) {
            return vr;
        }

        Map<Long, BaseStrategy> strategyById = StreamEx.of(vr.getValue())
                .map(CampaignWithPackageStrategy::getStrategyId)
                .distinct()
                .filter(strategyId -> strategyId != 0)
                .mapToEntry(container::getPackageStrategy)
                .toMap();

        Map<Long, Long> oneOfAlreadyLinkedToStrategyCampaignsIdByStrategyId = EntryStream.of(strategyById)
                .mapValues(s -> ((StrategyWithCampaignIds) s).getCids())
                .filterValues(CollectionUtils::isNotEmpty)
                .mapValues(cids -> cids.get(0))
                .toMap();

        Map<Long, BaseCampaign> linkedCampaignsById =
                campaignTypedRepository.getIdToModelTyped(
                        container.getShard(), oneOfAlreadyLinkedToStrategyCampaignsIdByStrategyId.values());

        Map<Long, CampaignType> linkedCampaignTypeById = EntryStream.of(linkedCampaignsById)
                .mapValues(c -> ((CampaignWithCampaignType) c).getType())
                .toMap();

        Map<Long, CampaignType> linkedCampaignsTypeByStrategyId =
                EntryStream.of(oneOfAlreadyLinkedToStrategyCampaignsIdByStrategyId)
                        .mapValues(linkedCampaignTypeById::get)
                        .nonNullValues()
                        .toMap();

        var validator = createValidator(strategyById, vr.getValue(), linkedCampaignsTypeByStrategyId);
        return new ListValidationBuilder<>(vr)
                .checkEachBy(validator)
                .getResult();
    }

    private Validator<ModelChanges<CampaignWithPackageStrategy>, Defect> createPreValidator(
            Map<Long, BaseStrategy> strategyFromModelChangesById,
            Map<Long, CampaignWithPackageStrategy> typesOfChangingCampaigns,
            Set<Long> idsOfCampaignsWithPublicStrategy,
            Map<Long, BaseStrategy> campaignIdToPackageStrategy) {
        return mc -> {
            Long strategyIdFromMc = mc.getPropIfChanged(CampaignWithPackageStrategy.STRATEGY_ID);

            BaseStrategy packageStrategy;

            if (strategyIdFromMc != null) {
                packageStrategy = strategyFromModelChangesById.get(strategyIdFromMc);
            } else {
                packageStrategy = campaignIdToPackageStrategy.get(mc.getId());
            }

            return new CampaignWithPackageStrategyUpdatePreValidator(
                    packageStrategy,
                    typesOfChangingCampaigns.get(mc.getId()),
                    idsOfCampaignsWithPublicStrategy.contains(mc.getId())).apply(mc);
        };
    }

    private Map<Long, CampaignWithPackageStrategy> getChangingCampaignsById(int shard,
                                                                            List<ModelChanges<CampaignWithPackageStrategy>> modelChanges) {
        var cids = mapList(modelChanges, ModelChanges::getId);
        return EntryStream.of(campaignTypedRepository.getIdToModelTyped(shard, cids))
                .mapValues(c -> (CampaignWithPackageStrategy) c)
                .toImmutableMap();
    }

    private static boolean isPublic(BaseStrategy strategy) {
        var isPublic = ((CommonStrategy) strategy).getIsPublic();
        return isPublic != null && isPublic;
    }

    private Validator<CampaignWithPackageStrategy, Defect> createValidator(Map<Long, BaseStrategy> strategyById,
                                                                           List<CampaignWithPackageStrategy> campaigns,
                                                                           Map<Long, CampaignType> linkedCampaignsTypesByStrategyId) {
        var maxNumberOfLinkedCids = ppcPropertiesSupport.get(MAX_NUMBER_OF_CIDS_ABLE_TO_LINK_TO_PACKAGE_STRATEGY)
                .getOrDefault(DEFAULT_MAX_NUMBERS_OF_CIDS_ABLE_TO_LINK_TO_PACKAGE_STRATEGY);
        return campaign ->
                new CampaignWithPackageStrategyUpdateValidator(
                        strategyById.get(campaign.getStrategyId()),
                        linkedCampaignsTypesByStrategyId.get(campaign.getStrategyId()), campaigns,
                        maxNumberOfLinkedCids).apply(campaign);
    }

    @Override
    public Class<CampaignWithPackageStrategy> getTypeClass() {
        return CampaignWithPackageStrategy.class;
    }
}
