package ru.yandex.direct.core.entity.campaign.service.validation.type.update;

import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.net.NetAcl;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.service.type.add.AddServicedCampaignService;
import ru.yandex.direct.core.entity.campaign.service.type.add.container.AddServicedCampaignInfo;
import ru.yandex.direct.core.entity.campaign.service.validation.type.CommonCampaignBeanValidatorContext;
import ru.yandex.direct.core.entity.campaign.service.validation.type.bean.CommonCampaignBeanValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.type.container.CampaignValidationContainer;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.requireServicingWillBeIgnored;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignValidators.dateNotInPastIfChanged;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignValidators.fieldCantBeModified;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.defect.CollectionDefects.duplicatedElement;

@Component
@ParametersAreNonnullByDefault
public class CommonCampaignUpdateValidationTypeSupport extends AbstractCampaignUpdateValidationTypeSupport<CommonCampaign> {
    private final NetAcl netAcl;
    private final AddServicedCampaignService addServicedCampaignService;

    @Autowired
    public CommonCampaignUpdateValidationTypeSupport(NetAcl netAcl,
                                                     AddServicedCampaignService addServicedCampaignService) {
        this.netAcl = netAcl;
        this.addServicedCampaignService = addServicedCampaignService;
    }

    @Override
    public Class<CommonCampaign> getTypeClass() {
        return CommonCampaign.class;
    }

    @Override
    public ValidationResult<List<ModelChanges<CommonCampaign>>, Defect> preValidate(
            CampaignValidationContainer container,
            ValidationResult<List<ModelChanges<CommonCampaign>>, Defect> vr) {
        var vb = new ListValidationBuilder<>(vr);

        vb.checkEach(unique(ModelChanges::getId), duplicatedElement(), When.isValid());

        vb.checkEachBy(fieldCantBeModified(CommonCampaign.CLIENT_ID));
        vb.checkEachBy(fieldCantBeModified(CommonCampaign.WALLET_ID));
        vb.checkEachBy(fieldCantBeModified(CommonCampaign.AGENCY_ID));
        vb.checkEachBy(fieldCantBeModified(CommonCampaign.AGENCY_UID));
        vb.checkEachBy(fieldCantBeModified(CommonCampaign.IS_SERVICE_REQUESTED));

        return vb.getResult();
    }

    @Override
    public ValidationResult<List<ModelChanges<CommonCampaign>>, Defect> validateBeforeApply(
            CampaignValidationContainer container,
            ValidationResult<List<ModelChanges<CommonCampaign>>, Defect> vr,
            Map<Long, CommonCampaign> unmodifiedModels) {
        new ListValidationBuilder<>(vr)
                .checkEachBy(dateNotInPastIfChanged(unmodifiedModels, CommonCampaign.START_DATE))
                .checkEachBy(dateNotInPastIfChanged(unmodifiedModels, CommonCampaign.END_DATE));
        return vr;
    }

    @Override
    public ValidationResult<List<CommonCampaign>, Defect> validate(
            CampaignValidationContainer container,
            ValidationResult<List<CommonCampaign>, Defect> vr) {
        var vb = new ListValidationBuilder<>(vr);

        CommonCampaignBeanValidatorContext context = new CommonCampaignBeanValidatorContext(netAcl);
        AddServicedCampaignInfo servicedInfo = addServicedCampaignService.getServicedInfoForClientCampaigns(container);
        vb.checkEachBy(x -> CommonCampaignBeanValidator.build(context,
                        container.getOptions().isLimitFioTo255Chars(),
                        container.getOptions().isValidateFioForForbiddenChars(),
                        container.getOptions().isValidateCampaignNameForForbiddenChars()
                ).apply(x))
                .weakCheckEach(requireServicingIsNotIgnored(container, servicedInfo.getIsServiced()));

        return vb.getResult();
    }

    private Constraint<CommonCampaign, Defect> requireServicingIsNotIgnored(
            CampaignValidationContainer container,
            Boolean isServiced) {
        return Constraint.fromPredicate(campaign -> {
            var notOk = container.getRequireServicing(campaign) == Boolean.TRUE
                    && isServiced == Boolean.FALSE;
            return !notOk;
        }, requireServicingWillBeIgnored());
    }
}
