package ru.yandex.direct.core.entity.campaignpayment.repository;

import java.math.BigDecimal;
import java.time.LocalDateTime;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static ru.yandex.direct.dbschema.ppc.tables.CampPaymentsInfo.CAMP_PAYMENTS_INFO;

@Repository
@ParametersAreNonnullByDefault
public class CampPaymentsInfoRepository {
    private final DslContextProvider dslContextProvider;

    @Autowired
    public CampPaymentsInfoRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Добавляет или обновляет информацию о платеже на кампанию
     *
     * @param shard          шард в который нужно записать информацию
     * @param campaignId     номер кампании
     * @param lastPaymentSum Количество денег с НДС, пришедших последним "платежём" (в валюте кампании)
     */
    public void addCampaignPayment(int shard, Long campaignId, BigDecimal lastPaymentSum) {
        addCampaignPayment(dslContextProvider.ppc(shard), campaignId, lastPaymentSum);
    }

    /**
     * Добавляет или обновляет информацию о платеже на кампанию
     * Версия для использования в рамках jooq-транзакций
     *
     * @param dslContext     инстанс {@link DSLContext}, информация будет записана в соответствующую ему БД
     * @param campaignId     номер кампании
     * @param lastPaymentSum Количество денег с НДС, пришедших последним "платежём" (в валюте кампании)
     */
    public void addCampaignPayment(DSLContext dslContext, Long campaignId, BigDecimal lastPaymentSum) {
        dslContext
                .insertInto(CAMP_PAYMENTS_INFO,
                        CAMP_PAYMENTS_INFO.CID, CAMP_PAYMENTS_INFO.LAST_PAYMENT_SUM,
                        CAMP_PAYMENTS_INFO.LAST_PAYMENT_TIME, CAMP_PAYMENTS_INFO.PAYMENTS_NUM)
                .values(campaignId, lastPaymentSum, LocalDateTime.now(), 1L)
                .onDuplicateKeyUpdate()
                .set(CAMP_PAYMENTS_INFO.PAYMENTS_NUM, CAMP_PAYMENTS_INFO.PAYMENTS_NUM.plus(1L))
                .set(CAMP_PAYMENTS_INFO.LAST_PAYMENT_TIME, MySQLDSL.values(CAMP_PAYMENTS_INFO.LAST_PAYMENT_TIME))
                .set(CAMP_PAYMENTS_INFO.LAST_PAYMENT_SUM, MySQLDSL.values(CAMP_PAYMENTS_INFO.LAST_PAYMENT_SUM))
                .execute();
    }
}
