package ru.yandex.direct.core.entity.campaignstatistic.repository

import org.jooq.impl.DSL
import org.springframework.stereotype.Repository
import ru.yandex.direct.core.entity.container.LocalDateRange
import ru.yandex.direct.grid.core.entity.model.GoalConversion
import ru.yandex.direct.grid.schema.yt.Tables.DIRECTGRIDGOALSSTAT_BS
import ru.yandex.direct.ytcomponents.service.CampaignGoalGridStatDynContextProvider
import ru.yandex.direct.ytwrapper.YtTableUtils
import ru.yandex.direct.ytwrapper.YtTableUtils.findColumnOrThrow
import ru.yandex.direct.ytwrapper.YtUtils.betweenRange
import ru.yandex.direct.ytwrapper.YtUtils.periodCondition
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL
import javax.annotation.ParametersAreNonnullByDefault

@Repository
@ParametersAreNonnullByDefault
class CampaignGoalsStatisticRepository(
    private val contextProvider: CampaignGoalGridStatDynContextProvider,
) {
    companion object {
        private val DIRECT_GOAL_STAT = DIRECTGRIDGOALSSTAT_BS.`as`("DirectGoalStat")
        private val GOALS_NUM =
            DSL.sum(DIRECT_GOAL_STAT.GOALS_NUM).`as`("GoalsNum")
        private val GOAL_ID = YtTableUtils.aliased(DIRECT_GOAL_STAT.GOAL_ID)
        private val EXPORT_ID = YtTableUtils.aliased(DIRECT_GOAL_STAT.EXPORT_ID)
    }

    fun getCampaignGoalsConversionsCountByCampaignId(
        localDateRange: LocalDateRange,
        goalIdsByCampaignId: Map<Long, Set<Long>>,
    ): Map<Long, List<GoalConversion>> {

        if (goalIdsByCampaignId.isEmpty()) {
            return emptyMap()
        }
        val goalIds = goalIdsByCampaignId.values.flatten()

        val query = YtDSL.ytContext()
            .select(EXPORT_ID, GOAL_ID, GOALS_NUM)
            .from(DIRECT_GOAL_STAT)
            .where(DIRECT_GOAL_STAT.GOAL_ID.`in`(goalIds)
                .and(DIRECT_GOAL_STAT.EXPORT_ID.`in`(goalIdsByCampaignId.keys))
                .and(periodCondition(DIRECT_GOAL_STAT.UPDATE_TIME, localDateRange.fromInclusive,
                    localDateRange.toInclusive)))
            .groupBy(EXPORT_ID, GOAL_ID)

        val rowset = contextProvider.getContext().executeSelect(query)

        return rowset.yTreeRows
            .filter {
                goalIdsByCampaignId.getValue(it.getOrThrow(EXPORT_ID.name).longValue())
                    .contains(it.getOrThrow(GOAL_ID.name).longValue())
            }
            .groupBy({ it.getOrThrow(EXPORT_ID.name).longValue() }, {
                GoalConversion()
                    .withGoalId(it.getOrThrow(GOAL_ID.name).longValue())
                    .withGoals(it.getOrThrow(GOALS_NUM.name).longValue())
            })
    }

    /**
     * Получить кол-во достижений указанных целей за период по указанным кампаниям
     *
     * @return кол-во достижений цели по ID цели
     */
    fun getGoalsConversionsCount(
        localDateRange: LocalDateRange,
        campaignIds: Collection<Long>,
        goalIds: Collection<GoalId>,
    ): Map<GoalId, Long> {
        if (campaignIds.isEmpty() || goalIds.isEmpty()) {
            return emptyMap()
        }
        val query = YtDSL.ytContext()
            .select(GOAL_ID, GOALS_NUM)
            .from(DIRECT_GOAL_STAT)
            .where(DIRECT_GOAL_STAT.GOAL_ID.`in`(goalIds)
                .and(DIRECT_GOAL_STAT.EXPORT_ID.`in`(campaignIds))
                .and(betweenRange(
                    DIRECT_GOAL_STAT.UPDATE_TIME,
                    localDateRange.fromInclusive,
                    localDateRange.toInclusive
                )))
            .groupBy(GOAL_ID)

        val rowset = contextProvider.getContext().executeSelect(query)

        val goalIdColIdx = findColumnOrThrow(rowset.schema, GOAL_ID)
        val goalsNumColIdx = findColumnOrThrow(rowset.schema, GOALS_NUM)
        return rowset.rows.asSequence()
            .map { it.values }
            .map { it[goalIdColIdx].longValue() to it[goalsNumColIdx].longValue() }
            .toMap()
    }
}

private typealias GoalId = Long
