package ru.yandex.direct.core.entity.campaignstatistic.repository;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.container.LocalDateRange;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.core.entity.model.GoalConversion;
import ru.yandex.direct.grid.core.entity.model.campaign.AggregatorGoal;

import static org.apache.commons.collections4.CollectionUtils.isNotEmpty;
import static ru.yandex.direct.core.entity.campaign.service.CampaignStrategyUtils.isAggregatorGoalId;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

@Service
@ParametersAreNonnullByDefault
public class CampaignStatisticService {
    private final CampaignGoalsStatisticRepository campaignGoalsStatisticRepository;
    private final CampaignService campaignService;

    @Autowired
    public CampaignStatisticService(CampaignGoalsStatisticRepository campaignGoalsStatisticRepository,
                                    CampaignService campaignService) {
        this.campaignGoalsStatisticRepository = campaignGoalsStatisticRepository;
        this.campaignService = campaignService;
    }

    public Map<Long, GoalConversion> getCampaignGoalConversionsCount(
            LocalDateRange localDateRange,
            Map<Long, Long> goalIdByCampaignId,
            Map<Long, List<AggregatorGoal>> aggregatorGoalsByCampaignId) {

        Map<Long, Set<Long>> goalIdsByCampaignId = EntryStream.of(goalIdByCampaignId)
                .mapToValue((campaignId, goalId) ->
                        extractGoalIdsFromAggregatedGoalId(aggregatorGoalsByCampaignId.get(campaignId), goalId))
                .toMap();

        var campaignGoalsConversionsCountByCampaignId =
                campaignGoalsStatisticRepository.getCampaignGoalsConversionsCountByCampaignId(
                        localDateRange, goalIdsByCampaignId);

        return EntryStream.of(goalIdByCampaignId)
                .mapToValue((campaignId, goalId) -> getGoalConversion(aggregatorGoalsByCampaignId,
                        campaignGoalsConversionsCountByCampaignId, campaignId, goalId))
                .nonNullValues()
                .toMap();

    }

    /**
     * Получить кол-во достижений указанных целей за период по всем кампаниям клиента
     *
     * @return кол-во достижений цели по ID цели
     */
    @NotNull
    public Map<Long, Long> getGoalsConversionsCount(ClientId clientId, Set<Long> goalIds,
                                                    LocalDateRange localDateRange) {
        Set<Long> clientCampaignIds = campaignService.getClientCampaignIds(clientId);
        Map<Long, Long> masterIdBySubId = campaignService.getSubCampaignIdsWithMasterIds(clientCampaignIds);
        Set<Long> statCampaignIds = Sets.union(clientCampaignIds, masterIdBySubId.keySet());

        return campaignGoalsStatisticRepository.getGoalsConversionsCount(
                localDateRange, statCampaignIds, goalIds);
    }

    private GoalConversion getGoalConversion(Map<Long, List<AggregatorGoal>> aggregatorGoalsByCampaignId,
                                             Map<Long, List<GoalConversion>> campaignGoalsConversionsCount,
                                             Long campaignId, Long goalId) {
        List<GoalConversion> goalConversions = campaignGoalsConversionsCount.get(campaignId);
        if (isNotEmpty(goalConversions)) {
            return isAggregatorGoalId(goalId)
                    ? getGoalConversionForAggregatedGoal(aggregatorGoalsByCampaignId.get(campaignId),
                    goalConversions, goalId)
                    : getGoalConversionForSimpleGoal(goalConversions, goalId);
        }
        return new GoalConversion().withGoalId(goalId).withGoals(0L);
    }

    private GoalConversion getGoalConversionForSimpleGoal(List<GoalConversion> goalsConversions, Long goalId) {
        return goalsConversions.stream()
                .filter(goal -> goalId.equals(goal.getGoalId()))
                .findAny().orElse(new GoalConversion()
                        .withGoalId(goalId)
                        .withGoals(0L));
    }

    private GoalConversion getGoalConversionForAggregatedGoal(
            List<AggregatorGoal> aggregatorGoals,
            List<GoalConversion> goalConversions,
            Long goalId) {

        Set<Long> subGoalIds = StreamEx.of(aggregatorGoals)
                .findAny(goal -> goal.getId().equals(goalId))
                .map(AggregatorGoal::getSubGoalIds)
                .map(Set::copyOf)
                .orElseThrow(() -> new IllegalStateException("Should exists sub goal ids for aggr goal id"));


        Long conversionCount = StreamEx.of(goalConversions)
                .mapToEntry(GoalConversion::getGoalId, GoalConversion::getGoals)
                .filterKeys(subGoalIds::contains)
                .values()
                .foldLeft(Long::sum)
                .orElse(0L);
        return new GoalConversion()
                .withGoalId(goalId)
                .withGoals(conversionCount);

    }

    private Set<Long> extractGoalIdsFromAggregatedGoalId(List<AggregatorGoal> aggregatorGoals,
                                                         Long goalId) {
        Map<Long, AggregatorGoal> aggregatorGoalById = listToMap(aggregatorGoals, AggregatorGoal::getId);
        return aggregatorGoalById != null && aggregatorGoalById.containsKey(goalId)
                ? Set.copyOf(aggregatorGoalById.get(goalId).getSubGoalIds())
                : Set.of(goalId);
    }
}
