package ru.yandex.direct.core.entity.campdaybudgethistory.repository;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;

import javax.annotation.Nullable;

import org.jooq.DSLContext;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.dbutil.SqlUtils;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static ru.yandex.direct.dbschema.ppc.tables.CampDayBudgetStopHistory.CAMP_DAY_BUDGET_STOP_HISTORY;
import static ru.yandex.direct.dbschema.ppc.tables.CampOptions.CAMP_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;

@Repository
public class CampDayBudgetStopHistoryRepository {

    private final DslContextProvider dslContextProvider;

    @Autowired
    public CampDayBudgetStopHistoryRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Возвращает устаревшие записи относительно {@param expireDateTime}.
     */
    public List<Long> getIdsOfExpiredRecords(int shard, LocalDateTime expireDateTime, int limit) {
        return getIdsOfExpiredRecords(dslContextProvider.ppc(shard), expireDateTime, limit);
    }

    public List<Long> getIdsOfExpiredRecords(DSLContext dslContext, LocalDateTime expireDateTime, int limit) {
        return dslContext
                .select(CAMP_DAY_BUDGET_STOP_HISTORY.ID)
                .from(CAMP_DAY_BUDGET_STOP_HISTORY)
                .where(CAMP_DAY_BUDGET_STOP_HISTORY.STOP_TIME.lessThan(expireDateTime))
                .limit(limit)
                .fetch(CAMP_DAY_BUDGET_STOP_HISTORY.ID);
    }

    /**
     * Удаляет записи по списку id {@param ids}.
     */
    public int delete(int shard, Collection<Long> ids) {
        return delete(dslContextProvider.ppc(shard), ids);
    }

    public int delete(DSLContext dslContext, Collection<Long> ids) {
        return dslContext
                .deleteFrom(CAMP_DAY_BUDGET_STOP_HISTORY)
                .where(CAMP_DAY_BUDGET_STOP_HISTORY.ID.in(ids))
                .execute();
    }


    /**
     * достает из базы время последней остановки кампании.
     * Проверяет что у кампании day_budget > 0 и day_budget_stop_time не 0000.00.00, и отдается null если это не так
     */
    @Nullable
    public LocalDateTime getDayBudgetStopHistoryForNotification(int shard, Long campaignId, Duration duration) {

        return dslContextProvider.ppc(shard)
                .select(CAMP_DAY_BUDGET_STOP_HISTORY.STOP_TIME)
                .from(CAMP_DAY_BUDGET_STOP_HISTORY)
                .join(CAMPAIGNS)
                .on(CAMPAIGNS.CID.eq(campaignId))
                .join(CAMP_OPTIONS)
                .on(CAMP_OPTIONS.CID.eq(campaignId))
                .where(CAMP_DAY_BUDGET_STOP_HISTORY.CID.eq(campaignId)
                        .and(CAMP_DAY_BUDGET_STOP_HISTORY.STOP_TIME
                                .greaterThan(SqlUtils.localDateTimeAdd(DSL.currentLocalDateTime(), duration.negated())))
                        .and(CAMPAIGNS.DAY_BUDGET.gt(BigDecimal.ZERO)))
                        .and(CAMP_OPTIONS.DAY_BUDGET_STOP_TIME.ne(SqlUtils.mysqlZeroLocalDateTime()))
                .groupBy(CAMP_DAY_BUDGET_STOP_HISTORY.STOP_TIME)
                .orderBy(CAMP_DAY_BUDGET_STOP_HISTORY.STOP_TIME.desc())
                .limit(1)
                .fetchOne(CAMP_DAY_BUDGET_STOP_HISTORY.STOP_TIME);
    }
}
