package ru.yandex.direct.core.entity.campoperationqueue;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.Iterables;
import one.util.streamex.StreamEx;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.campoperationqueue.model.CampQueueOperation;
import ru.yandex.direct.core.entity.campoperationqueue.model.CampQueueOperationName;
import ru.yandex.direct.dbschema.ppc.tables.records.CampOperationsQueueRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.write.JooqWriter;
import ru.yandex.direct.jooqmapper.write.JooqWriterBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.dbschema.ppc.tables.CampOperationsQueue.CAMP_OPERATIONS_QUEUE;
import static ru.yandex.direct.dbschema.ppc.tables.CampOperationsQueueCopy.CAMP_OPERATIONS_QUEUE_COPY;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromProperty;

@Repository
public class CampOperationQueueRepository {

    private final DslContextProvider dslContextProvider;

    private final JooqWriter<CampQueueOperation> jooqWriter;

    private static final int UPDATE_CHUNK_SIZE = 1000;

    public CampOperationQueueRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.jooqWriter = createJooqWriter();
    }

    public void addCampaignQueueOperations(int shard, Collection<CampQueueOperation> campQueueOperations) {
        if (isEmpty(campQueueOperations)) {
            return;
        }

        Iterable<List<CampQueueOperation>> chunks = Iterables.partition(campQueueOperations, UPDATE_CHUNK_SIZE);

        for (var chunk : chunks) {
            addCampaignQueueOperationsChunk(shard, chunk);
        }
    }

    public Set<Long> getCampaignIdsInQueueOperations(int shard, Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(CAMP_OPERATIONS_QUEUE.CID)
                .from(CAMP_OPERATIONS_QUEUE)
                .where(CAMP_OPERATIONS_QUEUE.CID.in(campaignIds))
                .fetchSet(CAMP_OPERATIONS_QUEUE.CID);
    }

    public Map<Long, Set<CampQueueOperationName>> getCampaignOperationNames(int shard, Collection<Long> campaignIds) {
        var result = dslContextProvider.ppc(shard)
                .select(CAMP_OPERATIONS_QUEUE.CID, CAMP_OPERATIONS_QUEUE.OPERATION)
                .from(CAMP_OPERATIONS_QUEUE)
                .where(CAMP_OPERATIONS_QUEUE.CID.in(campaignIds))
                .fetch();

        return StreamEx.of(result)
                .mapToEntry(
                        rec -> rec.get(CAMP_OPERATIONS_QUEUE.CID),
                        rec -> CampQueueOperationName.fromSource(rec.get(CAMP_OPERATIONS_QUEUE.OPERATION))
                )
                .groupingTo(HashSet::new);
    }

    private void addCampaignQueueOperationsChunk(int shard, List<CampQueueOperation> campQueueOperations) {
        InsertHelper<CampOperationsQueueRecord> insertHelper =
                new InsertHelper<>(dslContextProvider.ppc(shard), CAMP_OPERATIONS_QUEUE)
                        .addAll(jooqWriter, campQueueOperations)
                        .onDuplicateKeyIgnore();
        insertHelper.executeIfRecordsAdded();
    }

    public void deleteCampaignFromQueue(int shard, Collection<Long> campaignIds) {
        dslContextProvider.ppc(shard)
                .deleteFrom(CAMP_OPERATIONS_QUEUE)
                .where(CAMP_OPERATIONS_QUEUE.CID.in(campaignIds))
                .execute();
    }

    /**
     * @deprecated чтобы обратить внимание на отсутствие индекса
     */
    @Deprecated
    @SuppressWarnings("DeprecatedIsStillUsed")
    public void deleteCampaignFromCopyQueue(int shard, Collection<Long> campaignIds) {
        dslContextProvider.ppc(shard)
                .deleteFrom(CAMP_OPERATIONS_QUEUE_COPY)
                .where(CAMP_OPERATIONS_QUEUE_COPY.CID.in(campaignIds))
                .execute();
    }

    private JooqWriter<CampQueueOperation> createJooqWriter() {
        return JooqWriterBuilder.<CampQueueOperation>builder()
                .writeField(CAMP_OPERATIONS_QUEUE.CID, fromProperty(CampQueueOperation.CID))
                .writeField(CAMP_OPERATIONS_QUEUE.OPERATION,
                        fromProperty(CampQueueOperation.CAMP_QUEUE_OPERATION_NAME).by(CampQueueOperationName::toSource))
                .writeField(CAMP_OPERATIONS_QUEUE.PARAMS, fromProperty(CampQueueOperation.PARAMS))
                .build();
    }

}
