package ru.yandex.direct.core.entity.cashback.model;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;

import one.util.streamex.StreamEx;

public class CashbackCardCategoryInternal {
    private final Long id;
    private final String name;
    private final String description;
    private final String buttonText;
    private final String buttonLink;
    private BigDecimal maxPercent;
    private boolean isGeneral;
    private boolean isNew;
    private boolean isTechnical;
    private final List<CashbackCardProgramInternal> programs;

    public CashbackCardCategoryInternal(CashbackCategory dbCategory, String language) {
        this.id = dbCategory.getId();
        this.buttonLink = dbCategory.getButtonLink();
        this.programs = new ArrayList<>();
        switch (language) {
            case "RU":
                this.name = dbCategory.getNameRu();
                this.description = dbCategory.getDescriptionRu();
                this.buttonText = dbCategory.getButtonTextRu();
                break;
            case "EN":
                this.name = dbCategory.getNameEn();
                this.description = dbCategory.getDescriptionEn();
                this.buttonText = dbCategory.getButtonTextEn();
                break;
            default:
                throw new IllegalArgumentException(String.format("Unsupported language %s", language));
        }
    }

    public void addProgram(CashbackCardProgramInternal program) {
        programs.add(program);
    }

    public void prepareCategory() {
        programs.sort(Comparator.comparing(CashbackCardProgramInternal::getOrder));
        maxPercent = calculateMaxPercent();
        isGeneral = calculateIsGeneral();
        isNew = calculateIsNew();
        isTechnical = calculateIsTechnical();
    }

    public BigDecimal calculateMaxPercent() {
        return StreamEx.of(programs)
                .mapToEntry(CashbackCardProgramInternal::getOrder, CashbackCardProgramInternal::getPercent)
                .grouping(Collectors.maxBy(Comparator.naturalOrder()))
                .values()
                .stream()
                .map(o -> o.orElse(BigDecimal.ZERO))
                .reduce(BigDecimal.ZERO, BigDecimal::add);
    }

    public boolean calculateIsGeneral() {
        for (var program : programs) {
            if (!program.isGeneral()) {
                return false;
            }
        }
        return true;
    }

    public boolean calculateIsNew() {
        for (var program : programs) {
            if (program.isNew()) {
                return true;
            }
        }
        return false;
    }

    public boolean calculateIsTechnical() {
        for (var program : programs) {
            if (program.isTechnical()) {
                return true;
            }
        }
        return false;
    }

    public Long getId() {
        return id;
    }

    public String getName() {
        return name;
    }

    public String getDescription() {
        return description;
    }

    public BigDecimal getMaxPercent() {
        return maxPercent;
    }

    public String getButtonText() {
        return buttonText;
    }

    public String getButtonLink() {
        return buttonLink;
    }

    public List<CashbackCardProgramInternal> getPrograms() {
        return programs;
    }

    public boolean isNew() {
        return isNew;
    }

    public boolean isGeneral() {
        return isGeneral;
    }

    public boolean isTechnical() {
        return isTechnical;
    }
}
