package ru.yandex.direct.core.entity.cashback.repository;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.Field;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.cashback.model.CashbackProgramDetails;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplier;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder;

import static ru.yandex.direct.common.util.RepositoryUtils.booleanFromLong;
import static ru.yandex.direct.core.entity.cashback.CashbackConstants.PUBLIC_PROGRAMS_CLIENT_ID;
import static ru.yandex.direct.core.entity.cashback.CashbackConstants.PUBLIC_PROGRAMS_SHARD_ID;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_CASHBACK_DETAILS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_CASHBACK_PROGRAMS;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.utils.DateTimeUtils.atTheBeginningOfMonth;

@Repository
@ParametersAreNonnullByDefault
public class CashbackClientsRepository {
    private static final Field<BigDecimal> REWARD_SUM = DSL.sum(CLIENTS_CASHBACK_DETAILS.REWARD).as("reward_sum");

    private final ShardHelper shardHelper;
    private final DslContextProvider dslContextProvider;
    private final JooqReaderWithSupplier<CashbackProgramDetails> programDetailsMapper;

    @Autowired
    public CashbackClientsRepository(ShardHelper shardHelper, DslContextProvider dslContextProvider) {
        this.shardHelper = shardHelper;
        this.dslContextProvider = dslContextProvider;

        this.programDetailsMapper = JooqReaderWithSupplierBuilder.builder(CashbackProgramDetails::new)
                .readProperty(CashbackProgramDetails.PROGRAM_ID,
                        fromField(CLIENTS_CASHBACK_DETAILS.CASHBACK_PROGRAM_ID))
                .readProperty(CashbackProgramDetails.REWARD, fromField(CLIENTS_CASHBACK_DETAILS.REWARD))
                .readProperty(CashbackProgramDetails.REWARD_WITHOUT_NDS,
                        fromField(CLIENTS_CASHBACK_DETAILS.REWARD_WO_NDS))
                .readProperty(CashbackProgramDetails.DATE,
                        fromField(CLIENTS_CASHBACK_DETAILS.REWARD_DATE).by(LocalDateTime::toLocalDate))
                .build();
    }

    /**
     * Возвращает состояния (включена/выключена) программ для клиента по их идентификаторам
     */
    public Map<Long, Boolean> getClientProgramStates(ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return getClientProgramStates(shard, clientId);
    }

    public Map<Long, Boolean> getPublicProgramStates() {
        return getClientProgramStates(PUBLIC_PROGRAMS_SHARD_ID, PUBLIC_PROGRAMS_CLIENT_ID);
    }

    private Map<Long, Boolean> getClientProgramStates(int shard, ClientId clientId) {
        var raw = dslContextProvider.ppc(shard)
                .select(CLIENTS_CASHBACK_PROGRAMS.CASHBACK_PROGRAM_ID, CLIENTS_CASHBACK_PROGRAMS.IS_ENABLED)
                .from(CLIENTS_CASHBACK_PROGRAMS)
                .where(CLIENTS_CASHBACK_PROGRAMS.CLIENT_ID.eq(clientId.asLong()))
                .fetch();

        Map<Long, Boolean> result = new HashMap<>();
        raw.forEach(record -> result.put(record.value1(), booleanFromLong(record.value2())));
        return result;
    }

    /**
     * Возвращает список начислений кешбэков заданному клиенту за заданный период
     *
     * @param clientId идентификатор клиента
     * @param from     дата, от которой получить детализацию
     * @param to       дата, до которой получить деталлизацию
     */
    public List<CashbackProgramDetails> getProgramDetails(ClientId clientId, LocalDate from, LocalDate to) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        return dslContextProvider.ppc(shard)
                .select(programDetailsMapper.getFieldsToRead())
                .from(CLIENTS_CASHBACK_DETAILS)
                .where(CLIENTS_CASHBACK_DETAILS.CLIENT_ID.eq(clientId.asLong())
                        .and(CLIENTS_CASHBACK_DETAILS.REWARD_DATE.greaterOrEqual(from.atStartOfDay())
                                .and(CLIENTS_CASHBACK_DETAILS.REWARD_DATE.lessOrEqual(to.atStartOfDay()))))
                .orderBy(CLIENTS_CASHBACK_DETAILS.REWARD_DATE.asc())
                .fetch(programDetailsMapper::fromDb);
    }

    /**
     * Возвращает список начислений кешбэков заданному клиенту за выбранный месяц
     *
     * @param clientIds идентификаторы клиента
     * @param month     месяц, за который надо получить записи
     * @return
     */
    public Map<ClientId, List<CashbackProgramDetails>> getProgramDetails(Collection<ClientId> clientIds,
                                                                         LocalDate month) {
        var fieldsSetToRequest = programDetailsMapper.getFieldsToRead();
        fieldsSetToRequest.add(CLIENTS_CASHBACK_DETAILS.CLIENT_ID);
        return shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID, ClientId::asLong).stream()
                .mapKeyValue((shard, clientIdList) ->
                        dslContextProvider.ppc(shard)
                                .select(fieldsSetToRequest)
                                .from(CLIENTS_CASHBACK_DETAILS)
                                .where(CLIENTS_CASHBACK_DETAILS.CLIENT_ID.in(clientIdList)
                                        .and(CLIENTS_CASHBACK_DETAILS.REWARD_DATE.eq(atTheBeginningOfMonth(month)))
                                )
                                .fetch()
                ).flatMap(Collection::stream)
                .mapToEntry(record -> ClientId.fromLong(record.get(CLIENTS_CASHBACK_DETAILS.CLIENT_ID)),
                programDetailsMapper::fromDb)
                .sortedBy(Map.Entry::getKey)
                .collapseKeys()
                .toMap();
    }

    public Map<Long, BigDecimal> getRewardsSumByPrograms(ClientId clientId, Collection<Long> programIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        var raw = dslContextProvider.ppc(shard)
                .select(CLIENTS_CASHBACK_DETAILS.CASHBACK_PROGRAM_ID, REWARD_SUM)
                .from(CLIENTS_CASHBACK_DETAILS)
                .where(CLIENTS_CASHBACK_DETAILS.CLIENT_ID.eq(clientId.asLong())
                        .and(CLIENTS_CASHBACK_DETAILS.CASHBACK_PROGRAM_ID.in(programIds)))
                .groupBy(CLIENTS_CASHBACK_DETAILS.CASHBACK_PROGRAM_ID)
                .fetch();
        Map<Long, BigDecimal> result = new HashMap<>();
        raw.forEach(record -> result.put(record.value1(), record.value2()));
        return result;
    }

    /**
     * Возвращает список id включенных программ для клиента
     *
     * @param clientId идентификатор клиента
     */
    public List<Long> getClientPrograms(ClientId clientId) {
        Map<Long, Boolean> programStates = getClientProgramStates(clientId);
        return StreamEx.ofKeys(programStates, t -> t).toList();
    }

}
