package ru.yandex.direct.core.entity.cashback.repository;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;

import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.cashback.model.CashbackProgramCategory;
import ru.yandex.direct.core.entity.cashback.model.CashbackProgramCategoryExtended;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplier;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder;

import static ru.yandex.direct.dbschema.ppcdict.tables.CashbackCategories.CASHBACK_CATEGORIES;
import static ru.yandex.direct.dbschema.ppcdict.tables.CashbackPrograms.CASHBACK_PROGRAMS;
import static ru.yandex.direct.dbschema.ppcdict.tables.CashbackProgramsCategories.CASHBACK_PROGRAMS_CATEGORIES;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;

@Repository
public class CashbackProgramsCategoriesRepository {
    private final JooqReaderWithSupplier<CashbackProgramCategory> programCategoryMapper;
    private final JooqReaderWithSupplier<CashbackProgramCategoryExtended> programCategoryExtendedMapper;
    private final DslContextProvider dslContextProvider;

    public CashbackProgramsCategoriesRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;

        this.programCategoryMapper = JooqReaderWithSupplierBuilder.builder(CashbackProgramCategory::new)
                .readProperty(CashbackProgramCategory.ID, fromField(CASHBACK_PROGRAMS_CATEGORIES.LINK_ID))
                .readProperty(CashbackProgramCategory.PROGRAM_ID, fromField(
                        CASHBACK_PROGRAMS_CATEGORIES.CASHBACK_PROGRAM_ID))
                .readProperty(CashbackProgramCategory.CATEGORY_ID, fromField(
                        CASHBACK_PROGRAMS_CATEGORIES.CASHBACK_CATEGORY_ID))
                .readProperty(CashbackProgramCategory.ORDER, fromField(CASHBACK_PROGRAMS_CATEGORIES.ORDER))
                .build();
        this.programCategoryExtendedMapper = JooqReaderWithSupplierBuilder.builder(CashbackProgramCategoryExtended::new)
                .readProperty(CashbackProgramCategoryExtended.ID, fromField(CASHBACK_PROGRAMS_CATEGORIES.LINK_ID))
                .readProperty(CashbackProgramCategoryExtended.CATEGORY_ID,
                        fromField(CASHBACK_PROGRAMS_CATEGORIES.CASHBACK_CATEGORY_ID))
                .readProperty(CashbackProgramCategoryExtended.CATEGORY_NAME_RU, fromField(CASHBACK_CATEGORIES.NAME_RU))
                .readProperty(CashbackProgramCategoryExtended.PROGRAM_ID,
                        fromField(CASHBACK_PROGRAMS_CATEGORIES.CASHBACK_PROGRAM_ID))
                .readProperty(CashbackProgramCategoryExtended.PROGRAM_NAME_RU, fromField(CASHBACK_PROGRAMS.NAME_RU))
                .readProperty(CashbackProgramCategoryExtended.ORDER, fromField(CASHBACK_PROGRAMS_CATEGORIES.ORDER))
                .build();
    }

    @Nullable
    public CashbackProgramCategory getByIds(Long categoryId, Long programId) {
        return dslContextProvider.ppcdict()
                .select(programCategoryMapper.getFieldsToRead())
                .from(CASHBACK_PROGRAMS_CATEGORIES)
                .where(CASHBACK_PROGRAMS_CATEGORIES.CASHBACK_CATEGORY_ID.eq(categoryId))
                .and(CASHBACK_PROGRAMS_CATEGORIES.CASHBACK_PROGRAM_ID.eq(programId))
                .fetchOne(programCategoryMapper::fromDb);
    }

    public Map<Long, List<CashbackProgramCategory>> getCategoriesByPrograms(List<Long> programIds) {
        var data = dslContextProvider.ppcdict()
                .select(programCategoryMapper.getFieldsToRead())
                .from(CASHBACK_PROGRAMS_CATEGORIES)
                .where(CASHBACK_PROGRAMS_CATEGORIES.CASHBACK_PROGRAM_ID.in(programIds))
                .fetch(programCategoryMapper::fromDb);
        Map<Long, List<CashbackProgramCategory>> mapped = new HashMap<>();
        for (var dataRow : data) {
            if (!mapped.containsKey(dataRow.getProgramId())) {
                mapped.put(dataRow.getProgramId(), new ArrayList<>());
            }
            mapped.get(dataRow.getProgramId()).add(dataRow);
        }
        return mapped;
    }

    public List<CashbackProgramCategoryExtended> getAllExtended() {
        return dslContextProvider.ppcdict()
                .select(programCategoryExtendedMapper.getFieldsToRead())
                .from(CASHBACK_PROGRAMS_CATEGORIES)
                .join(CASHBACK_CATEGORIES)
                .on(CASHBACK_PROGRAMS_CATEGORIES.CASHBACK_CATEGORY_ID.eq(CASHBACK_CATEGORIES.CASHBACK_CATEGORY_ID))
                .join(CASHBACK_PROGRAMS)
                .on(CASHBACK_PROGRAMS_CATEGORIES.CASHBACK_PROGRAM_ID.eq(CASHBACK_PROGRAMS.CASHBACK_PROGRAM_ID))
                .fetch(programCategoryExtendedMapper::fromDb);
    }

    public CashbackProgramCategoryExtended getExtendedByIds(Long categoryId, Long programId) {
        return dslContextProvider.ppcdict()
                .select(programCategoryExtendedMapper.getFieldsToRead())
                .from(CASHBACK_PROGRAMS_CATEGORIES)
                .join(CASHBACK_CATEGORIES)
                .on(CASHBACK_PROGRAMS_CATEGORIES.CASHBACK_CATEGORY_ID.eq(CASHBACK_CATEGORIES.CASHBACK_CATEGORY_ID))
                .join(CASHBACK_PROGRAMS)
                .on(CASHBACK_PROGRAMS_CATEGORIES.CASHBACK_PROGRAM_ID.eq(CASHBACK_PROGRAMS.CASHBACK_PROGRAM_ID))
                .where(CASHBACK_PROGRAMS_CATEGORIES.CASHBACK_CATEGORY_ID.eq(categoryId))
                .and(CASHBACK_PROGRAMS_CATEGORIES.CASHBACK_PROGRAM_ID.eq(programId))
                .fetchOne(programCategoryExtendedMapper::fromDb);
    }
}
