package ru.yandex.direct.core.entity.cashback.repository;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.Configuration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.cashback.model.CashbackRewardDetails;
import ru.yandex.direct.core.entity.cashback.model.CashbackRewardDetailsRow;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_CASHBACK_DETAILS;
import static ru.yandex.direct.utils.DateTimeUtils.atTheBeginningOfMonth;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

@Repository
@ParametersAreNonnullByDefault
public class ImportCashbackRewardsRepository {

    private final ShardHelper shardHelper;
    private final DslContextProvider dslContextProvider;

    @Autowired
    public ImportCashbackRewardsRepository(ShardHelper shardHelper, DslContextProvider dslContextProvider) {
        this.shardHelper = shardHelper;
        this.dslContextProvider = dslContextProvider;
    }

    public void saveRewardDetails(Map<ClientId, CashbackRewardDetails> detailsByClients, LocalDate date) {
        saveRewardDetails(detailsByClients, date, false);
    }

    public void saveRewardDetails(Map<ClientId, CashbackRewardDetails> detailsByClients, LocalDate date,
                                  Boolean deleteOutdated) {
        var clientIds = detailsByClients.keySet();
        shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID).forEach((shard, clientIdsOnShard) -> {
            var detailsOnShard = listToMap(clientIdsOnShard, Function.identity(), detailsByClients::get);
            dslContextProvider.ppcTransaction(shard, conf -> {
                if (deleteOutdated) {
                    deleteOutdatedRewardDetails(conf,
                            StreamEx.of(detailsOnShard.keySet())
                                    .map(ClientId::asLong)
                                    .toList(),
                            date);
                }
                saveRewardDetails(conf, detailsOnShard, date);
            });
        });
    }

    private void deleteOutdatedRewardDetails(Configuration conf, List<Long> clientIds, LocalDate date) {
        conf.dsl().deleteFrom(CLIENTS_CASHBACK_DETAILS)
                .where(CLIENTS_CASHBACK_DETAILS.REWARD_DATE.eq(atTheBeginningOfMonth(date))
                        .and(CLIENTS_CASHBACK_DETAILS.CLIENT_ID.in(clientIds))
                ).execute();
    }

    private void saveRewardDetails(Configuration conf, Map<ClientId, CashbackRewardDetails> detailsByClients,
                                   LocalDate date) {
        var ids = generateDetailIds(detailsByClients);
        if (ids.isEmpty()) {
            return;
        }

        var idsIterator = ids.iterator();
        var insertStep = conf.dsl()
                .insertInto(CLIENTS_CASHBACK_DETAILS)
                .columns(CLIENTS_CASHBACK_DETAILS.CLIENT_CASHBACK_DETAILS_ID,
                        CLIENTS_CASHBACK_DETAILS.CASHBACK_PROGRAM_ID,
                        CLIENTS_CASHBACK_DETAILS.CLIENT_ID,
                        CLIENTS_CASHBACK_DETAILS.REWARD,
                        CLIENTS_CASHBACK_DETAILS.REWARD_WO_NDS,
                        CLIENTS_CASHBACK_DETAILS.REWARD_DATE);
        detailsByClients.forEach((clientId, details) -> {
            if (details.getDetails() == null) {
                return;
            }
            for (var row : details.getDetails()) {
                if (hasPositiveReward(row)) {
                    insertStep.values(idsIterator.next(),
                            row.getProgramId(),
                            clientId.asLong(),
                            row.getReward(),
                            row.getRewardWithoutNds(),
                            date.atStartOfDay());
                }
            }
        });
        insertStep.execute();
    }

    private List<Long> generateDetailIds(Map<ClientId, CashbackRewardDetails> detailsByClients) {
        var count = StreamEx.of(detailsByClients.values())
                .map(CashbackRewardDetails::getDetails)
                .nonNull()
                .flatMap(List::stream)
                .filter(this::hasPositiveReward)
                .count();
        return shardHelper.generateClientCashbackDetailsIds((int) count);
    }

    private boolean hasPositiveReward(CashbackRewardDetailsRow row) {
        return row.getReward().compareTo(BigDecimal.ZERO) > 0;
    }
}
