package ru.yandex.direct.core.entity.cashback.service;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.core.entity.cashback.model.CashbackCardsProgram;
import ru.yandex.direct.core.entity.cashback.model.CashbackClientInfo;
import ru.yandex.direct.core.entity.cashback.model.CashbackClientProgram;
import ru.yandex.direct.core.entity.cashback.model.CashbackProgram;
import ru.yandex.direct.core.entity.cashback.model.CashbackProgramDetails;
import ru.yandex.direct.core.entity.cashback.model.CashbackRewardsDetails;
import ru.yandex.direct.core.entity.cashback.repository.CashbackClientsRepository;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientNdsService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.currency.Percent;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.i18n.I18NBundle;

import static ru.yandex.direct.core.entity.cashback.CashbackConstants.TECHNICAL_PROGRAM_ID;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class CashbackClientsService {
    private final ClientNdsService clientNdsService;
    private final ClientService clientService;
    private final CashbackProgramsService cashbackProgramsService;
    private final CashbackClientsRepository cashbackClientsRepository;
    private final TranslationService translationService;

    @Autowired
    public CashbackClientsService(ClientNdsService clientNdsService,
                                  ClientService clientService,
                                  CashbackProgramsService cashbackProgramsService,
                                  CashbackClientsRepository cashbackClientsRepository,
                                  TranslationService translationService) {
        this.clientNdsService = clientNdsService;
        this.clientService = clientService;
        this.cashbackProgramsService = cashbackProgramsService;
        this.cashbackClientsRepository = cashbackClientsRepository;
        this.translationService = translationService;
    }

    public CashbackClientInfo getClientCashbackInfo(ClientId clientId) {
        var client = clientService.getClient(clientId);
        var consumedBonus = nvl(client.getCashBackBonus(), BigDecimal.ZERO);
        var awaitingBonus = nvl(client.getCashBackAwaitingBonus(), BigDecimal.ZERO);

        var lastMonthDetails = getPreviousMonthDetails(clientId);
        var lastMonthBonus = sumValues(lastMonthDetails, CashbackProgramDetails::getReward);
        var lastMonthBonusWithoutNds = sumValues(lastMonthDetails, CashbackProgramDetails::getRewardWithoutNds);

        var result = new CashbackClientInfo()
                .withAwaitingCashback(awaitingBonus)
                .withAwaitingCashbackWithoutNds(sumWithoutNds(client, awaitingBonus))
                .withLastMonthCashback(lastMonthBonus)
                .withLastMonthCashbackWithoutNds(lastMonthBonusWithoutNds)
                .withTotalCashback(consumedBonus)
                .withTotalCashbackWithoutNds(sumWithoutNds(client, consumedBonus));

        var clientsPrograms = cashbackProgramsService.getClientPrograms(clientId);
        var clientProgramsStates = cashbackClientsRepository.getClientProgramStates(clientId);
        var rewardsSumByPrograms = cashbackClientsRepository.getRewardsSumByPrograms(
                clientId, mapList(clientsPrograms, CashbackProgram::getId));
        var resultPrograms = mapList(clientsPrograms,
                program -> toClientProgram(program, clientProgramsStates, rewardsSumByPrograms));

        var technicalProgramState = clientProgramsStates.get(TECHNICAL_PROGRAM_ID);
        result.withPrograms(resultPrograms)
                .withCashbacksEnabled(technicalProgramState == null || technicalProgramState);

        return result;
    }

    public List<CashbackProgramDetails> getPreviousMonthDetails(ClientId clientId) {
        var previousMonthDate = LocalDate.now().minusMonths(1L);
        var dateFrom = previousMonthDate.withDayOfMonth(1);
        var dateTo = previousMonthDate.withDayOfMonth(previousMonthDate.lengthOfMonth());
        return cashbackClientsRepository.getProgramDetails(clientId, dateFrom, dateTo);
    }

    public CashbackRewardsDetails getClientCashbackRewardsDetails(ClientId clientId, int period) {
        var client = clientService.getClient(clientId);
        var consumedBonus = nvl(client.getCashBackBonus(), BigDecimal.ZERO);

        var dateFrom = LocalDate.now().minusMonths(period);
        var dateTo = LocalDate.now();
        var programDetails = cashbackClientsRepository.getProgramDetails(clientId, dateFrom, dateTo);
        var programIds = StreamEx.of(programDetails).map(CashbackProgramDetails::getProgramId).distinct().toList();
        var programInfoMap = cashbackProgramsService.getProgramsByIds(programIds);
        for (var program : programDetails) {
            if (programInfoMap.containsKey(program.getProgramId())) {
                program.withProgram(toCashbackCardsProgram(programInfoMap.get(program.getProgramId())));
            }
        }
        return new CashbackRewardsDetails()
                .withTotalCashback(consumedBonus)
                .withTotalCashbackWithoutNds(sumWithoutNds(client, consumedBonus))
                .withTotalByPrograms(programDetails);
    }

    /**
     * Включенность программы для конкретного пользователя. Считается по правилам:
     * <ul>
     * <li> Если техническая программа выключена, то все программы считаются выключенными
     * <li> Если программа публичная, то либо состояние программы для клиента отсутствует,
     *   либо программа для клиента включена (с учём включенности самой программы)
     * <li> Если программа не публичная, то она должна быть принудительно включена для клиента
     *      (с учётом включенности самой программы)
     * </ul>
     *
     * @param program Программа, статус которой вычисляем для пользователя
     * @param clientProgramsStates Состояния программ пользователя
     * @return {@code true}, если программа включена для клиента; иначе {@code false}
     */
    private static boolean isProgramEnabled(CashbackProgram program,
                                            Map<Long, Boolean> clientProgramsStates) {
        var technicalProgramState = clientProgramsStates.get(TECHNICAL_PROGRAM_ID);
        var isTechnicalProgramEnabled = technicalProgramState == null || technicalProgramState;

        if (!isTechnicalProgramEnabled) {
            return false;
        }
        var clientProgramState = clientProgramsStates.get(program.getId());
        if (program.getIsPublic()) {
            return program.getIsEnabled() && (clientProgramState == null || clientProgramState);
        } else {
            return program.getIsEnabled() && (clientProgramState != null && clientProgramState);
        }
    }

    private CashbackClientProgram toClientProgram(CashbackProgram program,
                                                  Map<Long, Boolean> clientProgramsStates,
                                                  Map<Long, BigDecimal> rewardsByPrograms) {
        var language = translationService.getLocale().getLanguage();
        var isRuLocale = I18NBundle.RU.getLanguage().equals(language);
        var programRewardsSum = rewardsByPrograms.get(program.getId());
        var isCategorySet = Objects.nonNull(program.getCategoryId());
        // Если к программе привязанна категория "по-старому", то в качестве названия и описания
        // программы используем название и описание категории.
        // Иначе используем название программы.
        String programName;
        String programDescription;
        if (isCategorySet) {
            programName = isRuLocale ? program.getCategoryNameRu() : program.getCategoryNameEn();
            programDescription = isRuLocale ? program.getCategoryDescriptionRu() : program.getCategoryDescriptionEn();
        } else {
            programName = isRuLocale ? program.getNameRu() : program.getNameEn();
            programDescription = "";
        }
        return new CashbackClientProgram()
                .withProgramId(program.getId())
                .withPercent(program.getPercent())
                .withName(programName)
                .withDescription(programDescription)
                .withEnabled(isProgramEnabled(program, clientProgramsStates))
                .withHasRewards(programRewardsSum != null && BigDecimal.ZERO.compareTo(programRewardsSum) <= 0)
                .withIsTechnical(program.getIsTechnical());
    }

    private BigDecimal sumWithoutNds(Client client, BigDecimal sum) {
        var currency = client.getWorkCurrency();
        var clientNds = clientNdsService.getClientNds(ClientId.fromLong(
                client.getAgencyClientId() != null && client.getAgencyClientId() > 0L ?
                        client.getAgencyClientId() : client.getClientId()
        ));
        var ndsPercent = clientNds == null ? null : clientNds.getNds();
        return subtractNds(sum, currency, ndsPercent);
    }

    public static BigDecimal subtractNds(BigDecimal sum, Currency currency, @Nullable Percent nds) {
        return subtractNds(sum, currency.getCode(), nds);
    }

    private static BigDecimal subtractNds(BigDecimal sum, CurrencyCode currency, @Nullable Percent nds) {
        var sumMoney = Money.valueOf(sum, currency);
        return nds == null ? sumMoney.bigDecimalValue() : sumMoney.subtractNds(nds).bigDecimalValue();
    }

    public static BigDecimal sumValues(List<CashbackProgramDetails> details,
                                        Function<CashbackProgramDetails, BigDecimal> valueToSumMapper) {
        return details
                .stream()
                .map(valueToSumMapper)
                .reduce(BigDecimal.ZERO, BigDecimal::add);
    }

    private static CashbackCardsProgram toCashbackCardsProgram(CashbackProgram program) {
        var isCategorySet = Objects.nonNull(program.getCategoryId());
        return new CashbackCardsProgram()
                .withId(program.getId())
                .withPercent(program.getPercent())
                .withIsTechnical(program.getIsTechnical())
                .withIsNew(program.getIsNew())
                .withIsGeneral(program.getIsPublic())
                .withNameRu(isCategorySet ? program.getCategoryNameRu() : program.getNameRu())
                .withNameEn(isCategorySet ? program.getCategoryNameEn() : program.getNameEn())
                .withTooltipInfoRu(program.getTooltipInfoRu())
                .withTooltipInfoEn(program.getTooltipInfoEn())
                .withTooltipLink(program.getTooltipLink())
                .withTooltipLinkTextRu(program.getTooltipLinkTextRu())
                .withTooltipLinkTextEn(program.getTooltipLinkTextEn());
    }
}
