package ru.yandex.direct.core.entity.cashback.service;

import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.cashback.model.CashbackProgram;
import ru.yandex.direct.core.entity.cashback.repository.CashbackClientsRepository;
import ru.yandex.direct.core.entity.cashback.repository.CashbackProgramsRepository;
import ru.yandex.direct.dbutil.model.ClientId;

import static ru.yandex.direct.core.entity.cashback.CashbackConstants.TECHNICAL_PROGRAM_ID;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;

@Service
public class CashbackProgramsService {
    private final CashbackProgramsRepository repository;
    private final CashbackClientsRepository clientsRepository;

    @Autowired
    public CashbackProgramsService(CashbackProgramsRepository repository, CashbackClientsRepository clientsRepository) {
        this.repository = repository;
        this.clientsRepository = clientsRepository;
    }

    public List<CashbackProgram> getAllPrograms() {
        return repository.get();
    }

    /**
     * Возвращает все программы, в которых участвует или участвовал пользователь
     * <p>
     * То есть будут возвращены:
     * <ul>
     *     <li>Все публичные включенные программы кроме технической</li>
     *     <li>Все публичные выключенные программы, для которых есть запись состояния у технического пользователя
     *         (то есть программа как минимум когда-то была включена)</li>
     *     <li>Все непубличные программы, для которых есть любая запись для заданного пользователя</li>
     * </ul>
     */
    public List<CashbackProgram> getClientPrograms(ClientId clientId) {
        var publicProgramsStates = clientsRepository.getPublicProgramStates();
        var clientProgramsStates = clientsRepository.getClientProgramStates(clientId);
        return filterList(getAllPrograms(), program -> {
            if (TECHNICAL_PROGRAM_ID.equals(program.getId())) {
                return false;
            }
            if (program.getIsPublic() && program.getIsEnabled()) {
                return true;
            } else if (program.getIsPublic()) {
                return publicProgramsStates.get(program.getId()) != null;
            }
            return clientProgramsStates.get(program.getId()) != null;
        });
    }

    public List<CashbackProgram> getProgramsByCategory(Long categoryId) {
        return repository.getByCategory(categoryId);
    }

    @Nullable
    public CashbackProgram getProgramById(Long programId) {
        return repository.get(programId);
    }

    public Map<Long, CashbackProgram> getProgramsByIds(List<Long> programIds) {
        return repository.getByIds(programIds);
    }
}
