package ru.yandex.direct.core.entity.cashback.service;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Consumer;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.cashback.model.CashbackRewardDetails;
import ru.yandex.direct.core.entity.cashback.model.CashbackRewardDetailsRow;
import ru.yandex.direct.core.entity.cashback.model.CashbackRewardTableRow;
import ru.yandex.direct.dbutil.model.ClientId;

import static ru.yandex.direct.utils.JsonUtils.fromJson;

@ParametersAreNonnullByDefault
public class CashbackRewardTableRowConsumer implements Consumer<CashbackRewardTableRow> {
    private static final Long MANUAL_CASHBACK_PROGRAM_ID = 14L;

    private final int directServiceId;
    private final Map<ClientId, CashbackRewardDetails> rewardsByClients;

    public CashbackRewardTableRowConsumer(int directServiceId) {
        this.directServiceId = directServiceId;
        this.rewardsByClients = new HashMap<>();
    }

    public Map<ClientId, CashbackRewardDetails> getRewardsByClients() {
        return rewardsByClients;
    }

    @Override
    public void accept(CashbackRewardTableRow row) {
        var serviceId = row.getServiceId();
        // В импорт попадают только записи для директа
        // В этих записях гарантировано currency=RUR (пока что)
        if (directServiceId == serviceId) {
            var details = Objects.nonNull(row.getCashbackDetails()) ? fromJson(row.getCashbackDetails(), CashbackRewardDetails.class) : new CashbackRewardDetails();
            if (Objects.isNull(details.getDetails())) {
                //Если строка без детализации, то записываем её в ручные кешбеки (programId=14)
                var detailsRow = new CashbackRewardDetailsRow()
                        .withProgramId(MANUAL_CASHBACK_PROGRAM_ID)
                        .withReward(new BigDecimal(row.getReward()))
                        .withRewardWithoutNds(new BigDecimal(row.getReward()).divide(new BigDecimal("1.2"),
                                RoundingMode.HALF_DOWN));
                List<CashbackRewardDetailsRow> manualDetails = new ArrayList<>();
                manualDetails.add(detailsRow);
                details = new CashbackRewardDetails().withDetails(manualDetails);
            }
            var id = ClientId.fromLong(row.getClientId());
            if (!rewardsByClients.containsKey(id)) {
                rewardsByClients.put(id, new CashbackRewardDetails().withDetails(new ArrayList<>()));
            }
            rewardsByClients.get(id).addRewardDetails(details);
        }
    }
}
