package ru.yandex.direct.core.entity.cashback.service;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.cashback.model.CashbackRewardTableRow;
import ru.yandex.direct.core.entity.cashback.model.CashbackRewardsImportParams;
import ru.yandex.direct.core.entity.cashback.repository.ImportCashbackRewardsRepository;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtTable;

@Service
public class ImportCashbackRewardsService {
    private static final Logger logger = LoggerFactory.getLogger(ImportCashbackRewardsService.class);

    private final YtProvider provider;
    private final ImportCashbackRewardsRepository rewardsRepository;

    private final int chunkSize;
    private final int directServiceId;

    @Autowired
    public ImportCashbackRewardsService(YtProvider provider,
                                        ImportCashbackRewardsRepository rewardsRepository,
                                        DirectConfig directConfig) {
        this.provider = provider;
        this.rewardsRepository = rewardsRepository;
        chunkSize = directConfig.getInt("cashbacks.yt.chunk_size");
        directServiceId = directConfig.getInt("balance.directServiceId");
    }

    public void importRewardsTables(List<CashbackRewardsImportParams> tablesParams) {
        for (var params : tablesParams) {
            importRewardsTable(params);
        }
    }

    public void importRewardsTable(CashbackRewardsImportParams params) {
        logger.info("Importing " + params.getCluster() + " / " + params.getTablePath());
        var table = new YtTable(params.getTablePath());

        var ytOperator = provider.getOperator(params.getCluster());

        for (long chunkIndex = 0; ; chunkIndex++) {
            logger.info("Chunk " + chunkIndex);
            var consumer = new CashbackRewardTableRowConsumer(directServiceId);
            ytOperator.readTableByRowRange(
                    table,
                    consumer,
                    new CashbackRewardTableRow(),
                    chunkSize * chunkIndex,
                    chunkSize * (chunkIndex + 1));

            var rewards = consumer.getRewardsByClients();
            if (rewards.isEmpty()) {
                break;
            }
            logger.info("Chunk size: " + rewards.size());
            rewardsRepository.saveRewardDetails(rewards, params.getDate(), true);
        }
    }
}
