package ru.yandex.direct.core.entity.changes.repository;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Record2;
import org.jooq.Result;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.changes.model.StatRollbackDropTypeRow;
import ru.yandex.direct.core.entity.statistics.StatRollbackInfo;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.utils.JsonUtils;

import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.STAT_ROLLBACKS;
import static ru.yandex.direct.dbschema.ppc.Tables.STAT_ROLLBACK_DATA;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
@ParametersAreNonnullByDefault
public class StatRollbacksRepository {

    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<StatRollbackInfo> statRollbacksDataMapper;


    @Autowired
    public StatRollbacksRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.statRollbacksDataMapper = createMapper();
    }

    private static JooqMapperWithSupplier<StatRollbackInfo> createMapper() {
        return JooqMapperWithSupplierBuilder.builder(StatRollbackInfo::new)
                .map(property(StatRollbackInfo.ID, STAT_ROLLBACK_DATA.ID))
                .map(property(StatRollbackInfo.UID, STAT_ROLLBACK_DATA.UID))
                .map(property(StatRollbackInfo.DATA, STAT_ROLLBACK_DATA.DATA))
                .build();
    }

    /**
     * Выбирать из списка кампаний такие, в которых были откаты по статистике с момента указанного времени.
     *
     * @param orderIds     список OrderID кампаний, которые нужно проверить
     * @param fromDateTime время, с момента которого нужно проверить, были ли изменения
     * @return список OrderID
     */
    public List<Long> getChangedOrderIds(List<Long> orderIds, LocalDateTime fromDateTime) {
        return dslContextProvider.ppcdict()
                .selectDistinct(STAT_ROLLBACKS.ORDER_ID)
                .from(STAT_ROLLBACKS)
                .where(STAT_ROLLBACKS.ORDER_ID.in(orderIds))
                .and(STAT_ROLLBACKS.ROLLBACK_TIME.ge(fromDateTime))
                .fetch(STAT_ROLLBACKS.ORDER_ID);
    }

    /**
     * Выбирать из списка кампаний такие, в которых были откаты по статистике с момента указанного времени.
     *
     * @param shard
     * @param campaignIds  список OrderID кампаний, которые нужно проверить
     * @param fromDateTime время, с момента которого нужно проверить, были ли изменения
     * @return мап CampaignID->BorderDate
     */
    public Map<Long, LocalDate> getChangedOrderIdsWithMinBorderDate(int shard, Collection<Long> campaignIds,
                                                                    LocalDateTime fromDateTime) {
        Result<Record2<Long, Long>> cidAndOrderIdresults = dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID, CAMPAIGNS.ORDER_ID)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.ORDER_ID.in(campaignIds))
                .and(CAMPAIGNS.ORDER_ID.isNotNull())
                .fetch();

        Map<Long, Long> campaignIdToOrderIdMap = cidAndOrderIdresults.stream()
                .collect(Collectors.toMap(Record2::value1, Record2::value2));

        Collection<Long> orderIds = campaignIdToOrderIdMap.values();

        Result<Record2<Long, LocalDate>> orderIdAndMinBorderDateResults = dslContextProvider.ppcdict()
                .select(STAT_ROLLBACKS.ORDER_ID, DSL.min(STAT_ROLLBACKS.BORDER_DATE))
                .from(STAT_ROLLBACKS)
                .where(STAT_ROLLBACKS.ORDER_ID.in(orderIds))
                .and(STAT_ROLLBACKS.ROLLBACK_TIME.ge(fromDateTime))
                .groupBy(STAT_ROLLBACKS.ORDER_ID)
                .fetch();

        Map<Long, LocalDate> orderIdToMinBorderDateMap = orderIdAndMinBorderDateResults.stream()
                .collect(Collectors.toMap(Record2::value1, Record2::value2));

        return cidAndOrderIdresults.stream()
                .filter(res -> orderIdToMinBorderDateMap.containsKey(res.value1()))
                .collect(Collectors.toMap(Record2::value1, res -> orderIdToMinBorderDateMap.get(res.value2())));
    }

    public Map<Long, Map<Long, StatRollbackDropTypeRow>> getStatRollbacksData(int shard, int limit) {
        List<StatRollbackInfo> rollbackInfos = dslContextProvider.ppc(shard)
                .select(STAT_ROLLBACK_DATA.ID, STAT_ROLLBACK_DATA.UID, STAT_ROLLBACK_DATA.DATA)
                .from(STAT_ROLLBACK_DATA)
                .limit(limit)
                .fetch()
                .map(statRollbacksDataMapper::fromDb);
        Map<Long, Map<Long, StatRollbackDropTypeRow>> result = new HashMap<>();
        for (StatRollbackInfo rollbackInfo : rollbackInfos) {
            StatRollbackDropTypeRow statRollbackDropTypeRow = JsonUtils.fromJson(rollbackInfo.getData(),
                    StatRollbackDropTypeRow.class);
            result.computeIfAbsent(rollbackInfo.getUid(), ignored -> new HashMap<>())
                    .put(rollbackInfo.getId(), statRollbackDropTypeRow);
        }
        return result;
    }

    public void deleteStatRollbacksDataRow(int shard, Long id) {
        dslContextProvider.ppc(shard)
                .deleteFrom(STAT_ROLLBACK_DATA)
                .where(STAT_ROLLBACK_DATA.ID.eq(id))
                .execute();
    }
}
