package ru.yandex.direct.core.entity.changes.service;

import java.time.LocalDateTime;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.campaign.model.CampaignSource;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampAggregatedLastchangeRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.changes.model.CheckCampaignsIntResp;
import ru.yandex.direct.core.entity.changes.repository.StatRollbacksRepository;
import ru.yandex.direct.dbutil.model.ClientId;

import static ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds.API5_CHANGES;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapAndFilterList;

@Service
@ParametersAreNonnullByDefault
public class CheckCampaignsService {
    private static final Logger logger = LoggerFactory.getLogger(CheckDictionariesService.class);

    private final CampAggregatedLastchangeRepository campAggregatedLastchangeRepository;
    private final StatRollbacksRepository statRollbacksRepository;
    private final AdGroupRepository adGroupRepository;
    private final BannerRelationsRepository bannerRelationsRepository;
    private final CampaignTypedRepository campaignTypedRepository;
    private final CampaignRepository campaignRepository;

    @Autowired
    public CheckCampaignsService(
            CampAggregatedLastchangeRepository campAggregatedLastchangeRepository,
            StatRollbacksRepository statRollbacksRepository,
            AdGroupRepository adGroupRepository,
            BannerRelationsRepository bannerRelationsRepository,
            CampaignTypedRepository campaignTypedRepository,
            CampaignRepository campaignRepository) {
        this.campAggregatedLastchangeRepository = campAggregatedLastchangeRepository;
        this.statRollbacksRepository = statRollbacksRepository;
        this.adGroupRepository = adGroupRepository;
        this.bannerRelationsRepository = bannerRelationsRepository;
        this.campaignTypedRepository = campaignTypedRepository;
        this.campaignRepository = campaignRepository;
    }

    /**
     * Получить список ID кампаний, в которых произошли изменения с момента времени requestTimestamp, с флагами
     * изменений.
     * <p>
     * Изменения бывают трёх видов:
     * <ol>
     *     <li>Изменения в самой кампании (SELF);</li>
     *     <li>Изменения в дочерних объектах кампании (CHILDREN);</li>
     *     <li>Изменения в статистике (STAT).</li>
     * </ol>
     *
     * @param clientId
     * @param shard
     * @param requestTimestamp с какого момента проверять изменения
     * @return
     */
    public List<CheckCampaignsIntResp> getCampaignsChanges(
            ClientId clientId,
            int shard,
            LocalDateTime requestTimestamp,
            boolean useCampAggregatedLastChange,
            Set<CampaignSource> visibleCampaignSources
    ) {
        // в общем методе показываем изменения в том числе и по удаленным кампаниям
        List<CommonCampaign> campaigns = campaignTypedRepository.getStrictly(shard,
                campaignRepository.getCampaignIdsByClientIds(shard, Set.of(clientId), API5_CHANGES),
                CommonCampaign.class);
        List<CommonCampaign> campaignsFiltered = campaigns.stream()
                .filter(c -> !c.getStatusEmpty())
                .filter(c -> !nvl(c.getIsUniversal(), false))
                .filter(c -> visibleCampaignSources.contains(nvl(c.getSource(), CampaignSource.DIRECT)))
                .collect(Collectors.toList());
        Set<Long> campaignIdsFiltered = listToSet(campaignsFiltered, CommonCampaign::getId);

        Set<Long> campaignIdsWhereChildrenWereChanged;
        if (useCampAggregatedLastChange) {
            campaignIdsWhereChildrenWereChanged =
                    campAggregatedLastchangeRepository.getChangedCampaignIds(shard, campaignIdsFiltered, requestTimestamp);
        } else {
            campaignIdsWhereChildrenWereChanged =
                    adGroupRepository.getChangedCampaignIds(shard, campaignIdsFiltered, requestTimestamp);
            Set<Long> campaignIdsWhereChildrenWereNotChanged = Sets.difference(campaignIdsFiltered,
                    campaignIdsWhereChildrenWereChanged);
            if (!campaignIdsWhereChildrenWereNotChanged.isEmpty()) {
                campaignIdsWhereChildrenWereChanged.addAll(
                        bannerRelationsRepository.getChangedCampaignIds(shard, campaignIdsWhereChildrenWereNotChanged,
                                requestTimestamp)
                );
            }
        }

        // поиск кампаний со STAT-изменениями с момента requestTimestamp (если orderId==0, то нет статистики)
        List<Long> orderIds = mapAndFilterList(campaignsFiltered, CommonCampaign::getOrderId,
                orderId -> orderId > 0);
        Set<Long> orderIdsWhereStatWasChanged = orderIds.isEmpty()
                ? Collections.emptySet()
                : new HashSet<>(statRollbacksRepository.getChangedOrderIds(orderIds, requestTimestamp));

        return mapAndFilterList(
                campaignsFiltered,
                campaign -> {
                    // проверить изменения SELF с момента requestTimestamp
                    boolean selfChanged = campaign.getLastChange() != null &&
                            requestTimestamp.isBefore(campaign.getLastChange());
                    boolean childrenChanged = campaignIdsWhereChildrenWereChanged.contains(campaign.getId());
                    boolean statChanged = orderIdsWhereStatWasChanged.contains(campaign.getOrderId());
                    return (selfChanged || childrenChanged || statChanged)
                            ? new CheckCampaignsIntResp(campaign.getId(), selfChanged, childrenChanged, statChanged)
                            : null;
                },
                Objects::nonNull
        );
    }
}
