package ru.yandex.direct.core.entity.changes.service;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerWithCampaignId;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType;
import ru.yandex.direct.core.entity.changes.model.CheckIntReq;
import ru.yandex.direct.core.entity.changes.model.CheckIntResp;
import ru.yandex.direct.core.entity.changes.repository.StatRollbacksRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.utils.CollectionUtils;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.changes.model.CheckIntResp.ParamBlock.ModifiedAdGroupIds;
import static ru.yandex.direct.core.entity.changes.model.CheckIntResp.ParamBlock.ModifiedAdIds;
import static ru.yandex.direct.core.entity.changes.model.CheckIntResp.ParamBlock.ModifiedCampaignIds;
import static ru.yandex.direct.core.entity.changes.model.CheckIntResp.ParamBlock.NotFoundAdGroupIds;
import static ru.yandex.direct.core.entity.changes.model.CheckIntResp.ParamBlock.NotFoundAdIds;
import static ru.yandex.direct.core.entity.changes.model.CheckIntResp.ParamBlock.NotFoundCampaignIds;
import static ru.yandex.direct.core.entity.changes.model.CheckIntResp.ParamBlock.UnprocessedAdGroupIds;
import static ru.yandex.direct.core.entity.changes.model.CheckIntResp.ParamBlock.UnprocessedCampaignIds;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.filterToSet;
import static ru.yandex.direct.utils.FunctionalUtils.flatMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class CheckService {

    private static final Logger logger = LoggerFactory.getLogger(CheckDictionariesService.class);

    private static final int ADGROUP_IDS_COUNT_QUERY_LIMIT = 10_001;
    private static final int AD_IDS_COUNT_QUERY_LIMIT = 50_001;

    private final CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory;
    private final ShardHelper shardHelper;
    private final CampaignRepository campaignRepository;
    private final AdGroupRepository adGroupRepository;
    private final BannerTypedRepository bannerTypedRepository;
    private final BannerRelationsRepository bannerRelationsRepository;
    private final StatRollbacksRepository statRollbacksRepository;

    @Autowired
    public CheckService(
            CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory,
            ShardHelper shardHelper,
            CampaignRepository campaignRepository,
            AdGroupRepository adGroupRepository,
            BannerTypedRepository bannerTypedRepository,
            BannerRelationsRepository bannerRelationsRepository,
            StatRollbacksRepository statRollbacksRepository
    ) {
        this.campaignSubObjectAccessCheckerFactory = campaignSubObjectAccessCheckerFactory;
        this.shardHelper = shardHelper;
        this.campaignRepository = campaignRepository;
        this.adGroupRepository = adGroupRepository;
        this.bannerTypedRepository = bannerTypedRepository;
        this.bannerRelationsRepository = bannerRelationsRepository;
        this.statRollbacksRepository = statRollbacksRepository;
    }

    /**
     * Главный диспетческий метод, определяющий какой из трёх нижеоуказанных методов вызвать.
     *
     * @param operatorUid
     * @param clientId
     * @param internalRequest
     * @return
     */
    public CheckIntResp processInternalRequest(Long operatorUid, ClientId clientId, CheckIntReq internalRequest) {
        CheckIntResp internalResponse = null;

        if (!CollectionUtils.isEmpty(internalRequest.getCampaignIds())) {
            internalResponse = getChangedCampaigns(operatorUid, clientId, internalRequest);

        } else if (!CollectionUtils.isEmpty(internalRequest.getAdGroupIds())) {
            internalResponse = getChangedAdGroups(operatorUid, clientId, internalRequest);

        } else if (!CollectionUtils.isEmpty(internalRequest.getAdIds())) {
            internalResponse = getChangedAds(operatorUid, clientId, internalRequest);
        }

        return internalResponse;
    }

    /**
     * Метод для ручки <a href="https://yandex.ru/dev/direct/doc/ref-v5/changes/check-docpage/#input">changes.check</a>
     * при поиске наличия изменений в <b>кампаниях</b> с указанной даты.
     */
    private CheckIntResp getChangedCampaigns(Long operatorUid, ClientId clientId, CheckIntReq internalRequest) {
        CheckIntResp result = new CheckIntResp();
        int shard = shardHelper.getShardByClientId(clientId);
        LocalDateTime fromTime = internalRequest.getFromTime();

        //  1. Убираем недоступные/несуществующие кампании из запроса, и кладём в блок NotFound
        //  2. Помешаем в респонс кампании, в которых были изменения, если нужно
        //  3. Помешаем в респонс кампании, в которых были изменения по статистике, если нужно
        //  4. Если запрашивались изменения в группах, то:
        //  4.1. Ищем группы по идентификаторам кампаний
        //  4.2. Помещаем в респонс те группы, в которых были изменения
        //  5. Если запрашивались изменения в объявлениях, то:
        //  5.1. Ищем объявления по идентификаторам кампаний
        //  5.2. Помещаем в респонс те объявления, в которых были изменения

        Set<Long> requestedCampaignIds = internalRequest.getCampaignIds();

        // разделяем список кампаний, на "доступные оператору" и "не найденные"
        Set<Long> accessibleCampaignIds = getAccessibleCampaignIds(operatorUid, clientId, requestedCampaignIds);

        // "недоступные из-за прав" - в ответе никак не выделяем, возвращаем как ненайденные
        Collection<Long> notFoundCampaignIds = org.apache.commons.collections.CollectionUtils.subtract(
                requestedCampaignIds, accessibleCampaignIds);
        if (org.apache.commons.collections.CollectionUtils.isNotEmpty(notFoundCampaignIds)) {
            result.setIds(NotFoundCampaignIds, notFoundCampaignIds);
        }

        boolean requestTimeIsFuture = internalRequest.getFromTime().isAfter(LocalDateTime.now());

        if (org.apache.commons.collections.CollectionUtils.isNotEmpty(accessibleCampaignIds) && !requestTimeIsFuture) {

            if (internalRequest.isSearchCampaignIdsFlag()) {
                result.setIds(ModifiedCampaignIds,
                        filterAndMapList(
                                campaignRepository.getCampaignsSimple(shard, accessibleCampaignIds).entrySet(),
                                e -> e.getValue().getLastChange() != null &&
                                        !e.getValue().getLastChange().isBefore(fromTime),
                                e -> e.getKey()
                        )
                );
            }

            if (internalRequest.isSearchCampaignStatFlag()) {
                result.setCampaignIdToBorderDateMap(
                        statRollbacksRepository
                                .getChangedOrderIdsWithMinBorderDate(shard, accessibleCampaignIds, fromTime)
                );
            }

            Long minUnprocessedCampaignId = null;
            if (internalRequest.isSearchAdGroupIdsFlag()) {
                Map<Long, List<Long>> adGroupIdsByCampaignIds = adGroupRepository.getChangedAdGroupIdsByCampaignIds(shard,
                        accessibleCampaignIds, fromTime, ADGROUP_IDS_COUNT_QUERY_LIMIT);
                minUnprocessedCampaignId = getMinUnprocessedCampaignId(adGroupIdsByCampaignIds, ADGROUP_IDS_COUNT_QUERY_LIMIT);
                Collection<Long> changedAdGroupIds;
                if (minUnprocessedCampaignId != null) {
                    changedAdGroupIds = getProcessingIds(adGroupIdsByCampaignIds, minUnprocessedCampaignId);
                }
                else {
                    changedAdGroupIds = flatMapToSet(adGroupIdsByCampaignIds.entrySet(), Map.Entry::getValue);
                }
                result.setIds(ModifiedAdGroupIds, changedAdGroupIds);
            }

            if (internalRequest.isSearchAdIdsFlag()) {
                Set<Long> processingCampaignIds = minUnprocessedCampaignId != null ?
                        getProcessingCampaignIds(accessibleCampaignIds, minUnprocessedCampaignId) : accessibleCampaignIds;
                Map<Long, List<Long>> adIdsByCampaignIds =
                        bannerRelationsRepository.getChangedBannersByCampaignIds(shard, processingCampaignIds,
                                fromTime, AD_IDS_COUNT_QUERY_LIMIT);
                Long minUnprocessedCampaignIdLocal = getMinUnprocessedCampaignId(adIdsByCampaignIds, AD_IDS_COUNT_QUERY_LIMIT);
                Collection<Long> changedAdIds;
                if (minUnprocessedCampaignIdLocal != null) {
                    changedAdIds = getProcessingIds(adIdsByCampaignIds, minUnprocessedCampaignIdLocal);
                    minUnprocessedCampaignId = minUnprocessedCampaignId != null && minUnprocessedCampaignId < minUnprocessedCampaignIdLocal ?
                            minUnprocessedCampaignId : minUnprocessedCampaignIdLocal;
                }
                else {
                    changedAdIds = flatMapToSet(adIdsByCampaignIds.entrySet(), Map.Entry::getValue);
                }
                result.setIds(ModifiedAdIds, changedAdIds);
            }

            if (minUnprocessedCampaignId != null) {
                result.setIds(UnprocessedCampaignIds, getUnprocessedCampaignIds(accessibleCampaignIds, minUnprocessedCampaignId));
            }
        }
        return result;
    }

    /**
     * Метод для ручки <a href="https://yandex.ru/dev/direct/doc/ref-v5/changes/check-docpage/#input">changes.check</a>
     * при поиске наличия изменений в <b>группах</b> с указанной даты.
     */
    private CheckIntResp getChangedAdGroups(Long operatorUid, ClientId clientId, CheckIntReq internalRequest) {
        CheckIntResp result = new CheckIntResp();
        int shard = shardHelper.getShardByClientId(clientId);
        LocalDateTime fromTime = internalRequest.getFromTime();

        //  1. По идентификаторам групп получаем идентификаторы кампаний
        //  2. Несуществующие ID помещаем в блок NotFound
        //  3. Отфильтровываем все недоступные кампании
        //  4. Идентификаторы групп, которые относятся к недоступным кампаниям, добавляем в блок NotFound
        //  5. Помешаем в респонс кампании, в которых были изменения (если запрашивалось)
        //  6. Помешаем в респонс кампании, в которых были изменения по статистике (если запрашивалось)
        //  7. Помешаем в респонс группы, в которых были изменения (если запрашивалось)
        //  8. Для запрошенных групп проверяем изменения в объявлениях и помешаем их в респонс (если запрашивалось)

        Set<Long> requestedAdGroupIds = internalRequest.getAdGroupIds();

        Map<Long, Long> campaignIdsByAdGroupIdsMap =
                adGroupRepository.getCampaignIdsByAdGroupIds(shard, requestedAdGroupIds);
        Set<Long> campaignIds = new HashSet<>(campaignIdsByAdGroupIdsMap.values());
        Set<Long> accessibleCampaignIds = getAccessibleCampaignIds(operatorUid, clientId, campaignIds);
        Set<Long> accessibleAdGroupIds = filterAndMapToSet(
                campaignIdsByAdGroupIdsMap.entrySet(),
                e -> accessibleCampaignIds.contains(e.getValue()),
                Map.Entry::getKey
        );

        Collection<Long> notFoundAdGroupIds = org.apache.commons.collections.CollectionUtils.subtract(requestedAdGroupIds, accessibleAdGroupIds);
        if (org.apache.commons.collections.CollectionUtils.isNotEmpty(notFoundAdGroupIds)) {
            result.setIds(NotFoundAdGroupIds, notFoundAdGroupIds);
        }

        boolean requestTimeIsFuture = internalRequest.getFromTime().isAfter(LocalDateTime.now());

        if (org.apache.commons.collections.CollectionUtils.isNotEmpty(accessibleAdGroupIds) && !requestTimeIsFuture) {
            if (internalRequest.isSearchCampaignIdsFlag()) {
                result.setIds(ModifiedCampaignIds,
                        filterAndMapList(
                                campaignRepository.getCampaignsSimple(shard, accessibleCampaignIds).entrySet(),
                                e -> e.getValue().getLastChange() != null &&
                                        !e.getValue().getLastChange().isBefore(fromTime),
                                e -> e.getKey()
                        )
                );
            }

            if (internalRequest.isSearchAdGroupIdsFlag()) {
                List<Long> changedAdGroupIds = filterAndMapList(
                        adGroupRepository.getAdGroups(shard, accessibleAdGroupIds),
                        adGroup -> adGroup.getLastChange() != null &&
                                !adGroup.getLastChange().isBefore(fromTime),
                        AdGroup::getId
                );
                result.setIds(ModifiedAdGroupIds, changedAdGroupIds);
            }

            if (internalRequest.isSearchAdIdsFlag()) {
                Map<Long, List<Long>> adIdsByCampaignIds =
                        bannerRelationsRepository.getChangedBannersByCampaignIds(shard, accessibleCampaignIds,
                                fromTime, AD_IDS_COUNT_QUERY_LIMIT);
                Long minUnprocessedCampaignId = getMinUnprocessedCampaignId(adIdsByCampaignIds, AD_IDS_COUNT_QUERY_LIMIT);
                Collection<Long> changedAdIds;
                if (minUnprocessedCampaignId != null) {
                    changedAdIds = getProcessingIds(adIdsByCampaignIds, minUnprocessedCampaignId);
                    Set<Long> unprocessedAdGroupIds = filterAndMapToSet(
                            campaignIdsByAdGroupIdsMap.entrySet(),
                            e -> accessibleCampaignIds.contains(e.getValue()) && e.getValue() >= minUnprocessedCampaignId,
                            Map.Entry::getKey
                    );
                    result.setIds(UnprocessedAdGroupIds, unprocessedAdGroupIds);
                }
                else {
                    changedAdIds = flatMapToSet(adIdsByCampaignIds.entrySet(), Map.Entry::getValue);
                }
                result.setIds(ModifiedAdIds, changedAdIds);
            }

            if (internalRequest.isSearchCampaignStatFlag()) {
                result.setCampaignIdToBorderDateMap(
                        statRollbacksRepository
                                .getChangedOrderIdsWithMinBorderDate(shard, accessibleCampaignIds, fromTime)
                );
            }
        }
        return result;
    }

    /**
     * Метод для ручки <a href="https://yandex.ru/dev/direct/doc/ref-v5/changes/check-docpage/#input">changes.check</a>
     * при поиске наличия изменений в <b>объявлениях</b> с указанной даты.
     */
    private CheckIntResp getChangedAds(Long operatorUid, ClientId clientId, CheckIntReq internalRequest) {
        CheckIntResp result = new CheckIntResp();
        int shard = shardHelper.getShardByClientId(clientId);
        LocalDateTime fromTime = internalRequest.getFromTime();

        //  1. Получаем объявления по запрошенным идентификаторам
        //  2. Несуществующие ID помещаем в блок NotFound
        //  3. Определяем идентификаторы кампаний и идентификаторы групп
        //  4. Отфильтровываем все недоступные кампании
        //  5. Идентификаторы объявлений, которые относятся к недоступным кампаниям, помещаем в блок NotFound
        //  6. Отфильтровываем объявления, чьи кампании отфильтровались
        //  7. Помешаем в респонс кампании, в которых были изменения (если запрашивалось)
        //  8. Помешаем в респонс кампании, в которых были изменения по статистике (если запрашивалось)
        //  9. Если запрашивались изменения в группах, то:
        //  9.1. отфильтровываем группы, относящиеся к недоступным кампаниям
        //  9.2. Помещаем в респонс те группы, в которых были изменения
        //  10. Помещаем в респонс идентификаторы баннеров, если в них были изменения (если запрашивалось)

        Set<Long> requestedAdIds = internalRequest.getAdIds();
        List<BannerWithCampaignId> banners =
                bannerTypedRepository.getSafely(shard, requestedAdIds, BannerWithCampaignId.class);

        Set<Long> campaignIds = new HashSet<>(mapList(banners, BannerWithCampaignId::getCampaignId));

        Set<Long> accessibleCampaignIds = getAccessibleCampaignIds(operatorUid, clientId, campaignIds);
        Set<Long> accessibleAdIds = filterAndMapToSet(
                banners,
                banner -> accessibleCampaignIds.contains(banner.getCampaignId()),
                Banner::getId
        );

        Collection<Long> notFoundAdIds = org.apache.commons.collections.CollectionUtils.subtract(requestedAdIds, accessibleAdIds);
        if (org.apache.commons.collections.CollectionUtils.isNotEmpty(notFoundAdIds)) {
            result.setIds(NotFoundAdIds, notFoundAdIds);
        }

        boolean requestTimeIsFuture = internalRequest.getFromTime().isAfter(LocalDateTime.now());

        if (org.apache.commons.collections.CollectionUtils.isNotEmpty(accessibleCampaignIds) && !requestTimeIsFuture) {

            if (internalRequest.isSearchCampaignIdsFlag()) {
                result.setIds(ModifiedCampaignIds,
                        filterAndMapList(
                                campaignRepository.getCampaignsSimple(shard, accessibleCampaignIds).entrySet(),
                                e -> e.getValue().getLastChange() != null &&
                                        !e.getValue().getLastChange().isBefore(fromTime),
                                e -> e.getKey()
                        )
                );
            }

            if (internalRequest.isSearchAdGroupIdsFlag()) {
                result.setIds(ModifiedAdGroupIds,
                        adGroupRepository.getChangedAdGroupIdsByCampaignIds(shard, accessibleCampaignIds, fromTime)
                );
            }

            if (internalRequest.isSearchAdIdsFlag()) {
                result.setIds(ModifiedAdIds,
                        bannerRelationsRepository.getChangedBannersByIds(shard, accessibleAdIds, fromTime)
                );
            }

            if (internalRequest.isSearchCampaignStatFlag()) {
                result.setCampaignIdToBorderDateMap(
                        statRollbacksRepository
                                .getChangedOrderIdsWithMinBorderDate(shard, accessibleCampaignIds, fromTime)
                );
            }

        }

        return result;
    }

    /**
     * Получить из заданного списка идентификаторов кампаний такие, к которым у оператора есть доступ.
     *
     * @param operatorUid
     * @param clientId
     * @param campaignIds Исходный список идентификаторов кампаний
     * @return
     */
    private Set<Long> getAccessibleCampaignIds(Long operatorUid, ClientId clientId, Set<Long> campaignIds) {
        CampaignSubObjectAccessValidator checker = campaignSubObjectAccessCheckerFactory
                .newCampaignChecker(operatorUid, clientId, campaignIds)
                .createValidator(CampaignAccessType.READ);

        return campaignIds.stream()
                .map(checker)
                .filter(vr -> !vr.hasAnyErrors())
                .map(ValidationResult::getValue)
                .collect(Collectors.toSet());
    }

    private Long getMinUnprocessedCampaignId(Map<Long, List<Long>> changedIdsByCampaignId, int limit) {
        if (changedIdsByCampaignId.values().stream().mapToInt(List::size).sum() == limit) {
            return Collections.max(changedIdsByCampaignId.keySet());
        }
        return null;
    }

    private Set<Long> getProcessingCampaignIds(Collection<Long> campaignIds, long minUnprocessedCampaignId) {
        return filterToSet(campaignIds, id -> id < minUnprocessedCampaignId);
    }

    private Set<Long> getProcessingIds(Map<Long, List<Long>> changedIdsByCampaignId, long minUnprocessedCampaignId) {
        return changedIdsByCampaignId.entrySet().stream()
            .filter(e -> !e.getKey().equals(minUnprocessedCampaignId))
            .flatMap(e -> e.getValue().stream())
            .collect(Collectors.toSet());
    }

    private Set<Long> getUnprocessedCampaignIds(Collection<Long> campaignIds, long minUnprocessedCampaignId) {
        return filterToSet(campaignIds, id -> id >= minUnprocessedCampaignId);
    }
}
