package ru.yandex.direct.core.entity.client.mcc

import org.jooq.impl.DSL
import org.springframework.stereotype.Repository
import ru.yandex.direct.dbschema.ppc.Tables.CLIENT_MCC_REQUESTS
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import java.time.LocalDateTime

@Repository
class ClientMccRepository(
    private val shardHelper: ShardHelper,
    private val dslContextProvider: DslContextProvider,
) {
    fun addRequest(shard: Int, controlClientId: ClientId, managedClientId: ClientId) {
        val requestId = shardHelper.generateClientMccRequestIds(1).first()
        dslContextProvider.ppc(shard)
            .insertInto(
                CLIENT_MCC_REQUESTS,
                CLIENT_MCC_REQUESTS.REQUEST_ID,
                CLIENT_MCC_REQUESTS.CLIENT_ID_FROM,
                CLIENT_MCC_REQUESTS.CLIENT_ID_TO
            )
            .values(requestId, controlClientId.asLong(), managedClientId.asLong())
            .onDuplicateKeyUpdate().set(CLIENT_MCC_REQUESTS.CREATE_TIME, LocalDateTime.now())
            .execute()
    }

    fun deleteRequest(shard: Int, controlClientId: ClientId, managedClientId: ClientId): Int {
        return dslContextProvider.ppc(shard)
            .deleteFrom(CLIENT_MCC_REQUESTS)
            .where(CLIENT_MCC_REQUESTS.CLIENT_ID_FROM.eq(controlClientId.asLong())
                .and(CLIENT_MCC_REQUESTS.CLIENT_ID_TO.eq(managedClientId.asLong())))
            .execute()
    }

    fun deleteRequest(shard: Int, requestId: Long) {
        dslContextProvider.ppc(shard)
            .deleteFrom(CLIENT_MCC_REQUESTS)
            .where(CLIENT_MCC_REQUESTS.REQUEST_ID.eq(requestId))
            .execute()
    }

    fun getRequestByClientIdPair(shard: Int, controlClientId: ClientId, managedClientId: ClientId): ClientMccRequest? {
        return dslContextProvider.ppc(shard)
            .select(
                CLIENT_MCC_REQUESTS.REQUEST_ID,
                CLIENT_MCC_REQUESTS.CLIENT_ID_FROM,
                CLIENT_MCC_REQUESTS.CLIENT_ID_TO,
                CLIENT_MCC_REQUESTS.CREATE_TIME
            )
            .from(CLIENT_MCC_REQUESTS)
            .where(CLIENT_MCC_REQUESTS.CLIENT_ID_FROM.eq(controlClientId.asLong())
                .and(CLIENT_MCC_REQUESTS.CLIENT_ID_TO.eq(managedClientId.asLong())))
            .fetchInto(CLIENT_MCC_REQUESTS)
            .map(ClientMccRequest::fromRecord)
            .firstOrNull()
    }

    fun getRequestsByControlClientId(shard: Int, controlClientId: ClientId): Set<ClientMccRequest> {
        return dslContextProvider.ppc(shard)
            .select(
                CLIENT_MCC_REQUESTS.REQUEST_ID,
                CLIENT_MCC_REQUESTS.CLIENT_ID_FROM,
                CLIENT_MCC_REQUESTS.CLIENT_ID_TO,
                CLIENT_MCC_REQUESTS.CREATE_TIME
            )
            .from(CLIENT_MCC_REQUESTS)
            .where(CLIENT_MCC_REQUESTS.CLIENT_ID_FROM.eq(controlClientId.asLong()))
            .fetchInto(CLIENT_MCC_REQUESTS)
            .map(ClientMccRequest::fromRecord)
            .toSet()
    }

    fun hasRequestsByControlClientId(shard: Int, controlClientId: ClientId): Boolean {
        return dslContextProvider.ppc(shard)
            .select(DSL.value(1))
            .from(CLIENT_MCC_REQUESTS)
            .where(CLIENT_MCC_REQUESTS.CLIENT_ID_FROM.eq(controlClientId.asLong()))
            .limit(1)
            .fetchAny() != null
    }

    fun getRequestsByManagedClientId(shard: Int, managedClientId: ClientId): Set<ClientMccRequest> {
        return dslContextProvider.ppc(shard)
            .select(
                CLIENT_MCC_REQUESTS.REQUEST_ID,
                CLIENT_MCC_REQUESTS.CLIENT_ID_FROM,
                CLIENT_MCC_REQUESTS.CLIENT_ID_TO,
                CLIENT_MCC_REQUESTS.CREATE_TIME
            )
            .from(CLIENT_MCC_REQUESTS)
            .where(CLIENT_MCC_REQUESTS.CLIENT_ID_TO.eq(managedClientId.asLong()))
            .fetchInto(CLIENT_MCC_REQUESTS)
            .map(ClientMccRequest::fromRecord)
            .toSet()
    }
}
