package ru.yandex.direct.core.entity.client.mcc

import org.springframework.stereotype.Service
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.rbac.RbacClientsRelations
import ru.yandex.direct.rbac.RbacService

/**
 * Сервис для работы с клиентским МСС
 */
@Service
class ClientMccService(
    private val clientMccRepository: ClientMccRepository,
    private val rbacClientsRelations: RbacClientsRelations,
    private val rbacService: RbacService,
    private val shardHelper: ShardHelper,
) {
    /**
     * Создаем заявку на управление МСС
     *
     * @param controlClientId id управляющего аккаунта МСС
     * @param managedClientId id клиента, который подал заявку на управление
     */
    fun addRequest(controlClientId: ClientId, managedClientId: ClientId) {
        val shard = shardHelper.getShardByClientId(controlClientId)
        return clientMccRepository.addRequest(shard, controlClientId, managedClientId)
    }

    /**
     * Удаляем заявку на управление МСС
     *
     * @param controlClientId id управляющего аккаунта МСС
     * @param managedClientId id клиента, который подал заявку на управление
     */
    fun deleteRequest(controlClientId: ClientId, managedClientId: ClientId): Boolean {
        val shard = shardHelper.getShardByClientId(controlClientId)
        val result = clientMccRepository.deleteRequest(shard, controlClientId, managedClientId)
        return result > 0
    }

    /**
     * Подтверждаем заявку на управление МСС
     *
     * @param controlClientId id управляющего аккаунта МСС
     * @param managedClientId id клиента, который подал заявку на управление
     */
    fun approveRequest(controlClientId: ClientId, managedClientId: ClientId): Boolean {
        val shard = shardHelper.getShardByClientId(controlClientId)
        val request = clientMccRepository.getRequestByClientIdPair(shard, controlClientId, managedClientId)
            ?: return false
        rbacClientsRelations.addClientMccRelation(controlClientId, managedClientId)
        clientMccRepository.deleteRequest(shard, request.requestId!!)
        return true
    }

    /**
     * Отклоняем заявку на управление МСС
     *
     * @param controlClientId id управляющего аккаунта МСС
     * @param managedClientId id клиента, который подал заявку на управление
     */
    fun declineRequest(controlClientId: ClientId, managedClientId: ClientId): Boolean {
        return deleteRequest(controlClientId, managedClientId)
    }

    /**
     * Удалить связь клиента с МСС
     *
     * @param controlClientId id управляющего аккаунта МСС
     * @param managedClientId id клиента, к которому будет доступ у управляющего аккаунта
     */
    fun unlinkMcc(controlClientId: ClientId, managedClientId: ClientId): Boolean {
        return rbacClientsRelations.removeClientMccRelation(controlClientId, managedClientId)
    }

    /**
     * Получаем список идентификаторов управляющих аккаунтов для заданного клиента
     *
     * @param managedClientId id клиента, для которого получаем управляющие аккакунты
     */
    fun getControlClients(managedClientId: ClientId): Set<ClientId> {
        val controlMccRelations = rbacClientsRelations.getControlMccRelations(managedClientId)
        return controlMccRelations.map { ClientId.fromLong(it.clientIdFrom) }.toSet()
    }

    /**
     * Получаем список идентификаторов управляемых аккаунтов для заданного управляющего аккаунта
     *
     * @param controlClientId id клиента, для которого получаем управляемые аккакунты
     */
    fun getManagedClients(controlClientId: ClientId): Set<ClientId> {
        return rbacClientsRelations.getMangedMccClientIds(controlClientId)
    }

    /**
     * Получаем все заявки на управление, которые создал заданный клиент
     *
     * @param managedClientId id клиента, заявки которого получаем
     */
    fun getOwnRequests(managedClientId: ClientId): Set<ClientMccRequest> {
        val requests = mutableSetOf<ClientMccRequest>()
        shardHelper.forEachShard { shard ->
            requests.addAll(clientMccRepository.getRequestsByManagedClientId(shard, managedClientId))
        }
        return requests
    }

    /**
     * Получаем все заявки на управление для заданного управляющего аккаунта
     *
     * @param managedClientId id управляющего аккаунта
     */
    fun getControlRequests(controlClientId: ClientId): Set<ClientMccRequest> {
        val shard = shardHelper.getShardByClientId(controlClientId)
        return clientMccRepository.getRequestsByControlClientId(shard, controlClientId)
    }

    /**
     * Получаем флаг наличия заявок на управление для заданного управляющего аккаунта
     *
     * @param managedClientId id управляющего аккаунта
     */
    fun hasControlRequests(controlClientId: ClientId): Boolean {
        val shard = shardHelper.getShardByClientId(controlClientId)
        return clientMccRepository.hasRequestsByControlClientId(shard, controlClientId)
    }

    /**
     * Получаем флаг наличия управляемых клиентов для заданного управляющего аккаунта
     *
     * @param uid идентификатор главного представителя управляющего аккаунта
     */
    fun hasManagedClients(uid: Long): Boolean {
        return rbacService.isUserMccControlClient(uid)
    }
}
